<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Core\Auth;
use App\Core\Csrf;
use App\Core\DB;
use App\Core\Tenant;

final class DeleteRequestsController
{
  public function request(): void
  {
    Csrf::verify($_POST['csrf'] ?? null);

    $pdo = DB::pdo();
    $companyId = Tenant::companyId();
    $u = Auth::user();

    $entity = (string)($_POST['entity'] ?? '');
    $entityId = (int)($_POST['entity_id'] ?? 0);
    $reason = trim((string)($_POST['reason'] ?? ''));

    if (!in_array($entity, ['document','customer'], true) || $entityId <= 0) {
      http_response_code(422); echo "Invalid"; return;
    }

    // staff zaten sadece talep açar; admin de isterse talep açabilir (opsiyonel)
    $st = $pdo->prepare("
      INSERT INTO delete_requests (company_id, requested_by, entity, entity_id, reason, status)
      VALUES (?,?,?,?,?, 'pending')
    ");
    $st->execute([$companyId, (int)$u['id'], $entity, $entityId, $reason]);

    Auth::redirect('/docs'); // örnek dönüş
  }

  public function list(): void
  {
    $pdo = DB::pdo();
    $companyId = Tenant::companyId();

    $st = $pdo->prepare("
      SELECT dr.*, u.name AS requested_by_name
      FROM delete_requests dr
      JOIN users u ON u.id = dr.requested_by
      WHERE dr.company_id=? AND dr.status='pending'
      ORDER BY dr.requested_at ASC
    ");
    $st->execute([$companyId]);
    $rows = $st->fetchAll();

    // Basit çıktı (view'e bağlayabilirsin)
    header('Content-Type: text/plain; charset=utf-8');
    foreach ($rows as $r) {
      echo "#{$r['id']} {$r['entity']} {$r['entity_id']} by {$r['requested_by_name']}\n";
    }
  }

  public function decide(): void
  {
    Csrf::verify($_POST['csrf'] ?? null);

    $pdo = DB::pdo();
    $companyId = Tenant::companyId();
    $admin = Auth::user();

    $reqId = (int)($_POST['request_id'] ?? 0);
    $decision = (string)($_POST['decision'] ?? 'rejected'); // approved/rejected

    if ($reqId <= 0 || !in_array($decision, ['approved','rejected'], true)) {
      http_response_code(422); echo "Invalid"; return;
    }

    try {
      $pdo->beginTransaction();

      $st = $pdo->prepare("SELECT * FROM delete_requests WHERE id=? AND company_id=? FOR UPDATE");
      $st->execute([$reqId, $companyId]);
      $req = $st->fetch();
      if (!$req || $req['status'] !== 'pending') throw new \RuntimeException("Not found/pending");

      if ($decision === 'approved') {
        $this->applySoftDelete($pdo, $companyId, (string)$req['entity'], (int)$req['entity_id']);
      }

      $up = $pdo->prepare("
        UPDATE delete_requests
        SET status=?, approved_by=?, decided_at=NOW()
        WHERE id=? AND company_id=?
      ");
      $up->execute([$decision, (int)$admin['id'], $reqId, $companyId]);

      $pdo->commit();
      Auth::redirect('/delete-requests');
    } catch (\Throwable $e) {
      if ($pdo->inTransaction()) $pdo->rollBack();
      http_response_code(500);
      echo "Decision error: " . htmlspecialchars($e->getMessage());
    }
  }

  private function applySoftDelete(\PDO $pdo, int $companyId, string $entity, int $entityId): void
  {
    if ($entity === 'document') {
      $st = $pdo->prepare("UPDATE invoices_quotations SET deleted_at=NOW() WHERE id=? AND company_id=?");
      $st->execute([$entityId, $companyId]);
    } elseif ($entity === 'customer') {
      $st = $pdo->prepare("UPDATE customers SET deleted_at=NOW() WHERE id=? AND company_id=?");
      $st->execute([$entityId, $companyId]);
    } else {
      throw new \RuntimeException("Unsupported entity");
    }
  }
}
