<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Core\Auth;
use App\Core\Csrf;
use App\Core\DB;
use App\Core\Tenant;
use App\Core\Security;

final class DocumentsController
{
  // GET /docs/create
  public function createForm(): void
  {
    $pdo = DB::pdo();
    $cid = Tenant::companyId();

    $st = $pdo->prepare("SELECT id,name FROM customers WHERE company_id=? AND deleted_at IS NULL ORDER BY name");
    $st->execute([$cid]);
    $customers = $st->fetchAll();

    $title = "Belge Oluştur";

    ob_start();
    require __DIR__ . '/../Views/documents/create_form_inner.php';
    $content = ob_get_clean();

    require __DIR__ . '/../Views/layouts/main.php';
  }

  // POST /docs/create
  public function createSave(): void
  {
    Csrf::verify($_POST['csrf'] ?? null);

    $pdo = DB::pdo();
    $companyId = Tenant::companyId();
    $user = Auth::user();
    $userId = (int)($user['id'] ?? 0);

    $type = (string)($_POST['type'] ?? 'invoice');
    if (!in_array($type, ['invoice','quote'], true)) $type = 'invoice';

    $customerId = (int)($_POST['customer_id'] ?? 0);
    $issueDate  = (string)($_POST['issue_date'] ?? date('Y-m-d'));
    $dueDate    = (string)($_POST['due_date'] ?? '');
    $currency   = (string)($_POST['currency'] ?? 'TRY');
    $notes      = (string)($_POST['notes'] ?? '');

    $items = $_POST['items'] ?? [];
    if ($customerId <= 0 || !is_array($items) || count($items) === 0) {
      http_response_code(422);
      echo "Invalid input";
      return;
    }

    // Kalemleri temizle + hesapla
    $subtotal = 0.0;
    $totalTax = 0.0;
    $cleanItems = [];
    $lineNo = 1;

    foreach ($items as $it) {
      $title = trim((string)($it['title'] ?? ''));
      if ($title === '') continue;

      $qty   = (float)($it['quantity'] ?? 1);
      $price = (float)($it['unit_price'] ?? 0);
      $vat   = (float)($it['vat_rate'] ?? 20);

      if ($qty <= 0) $qty = 1;
      if ($price < 0) $price = 0;
      if ($vat < 0) $vat = 0;

      $lineBase  = $qty * $price;
      $vatAmount = round($lineBase * ($vat/100), 2);
      $lineTotal = round($lineBase + $vatAmount, 2);

      $subtotal += $lineBase;
      $totalTax += $vatAmount;

      $cleanItems[] = [
        'line_no' => $lineNo++,
        'title' => $title,
        'description' => trim((string)($it['description'] ?? '')),
        'quantity' => $qty,
        'unit' => trim((string)($it['unit'] ?? '')),
        'unit_price' => $price,
        'vat_rate' => $vat,
        'vat_amount' => $vatAmount,
        'line_total' => $lineTotal,
      ];
    }

    if (count($cleanItems) === 0) {
      http_response_code(422);
      echo "No valid items";
      return;
    }

    $totalAmount = round($subtotal + $totalTax, 2);

    // Doc No üretimi (basit)
    $prefix = $type === 'invoice' ? 'INV' : 'QUO';
    $docNo = $prefix . '-' . date('Y') . '-' . str_pad((string)random_int(1, 9999), 4, '0', STR_PAD_LEFT);

    try {
      $pdo->beginTransaction();

      // Customer doğrulama (company scope)
      $chk = $pdo->prepare("SELECT id FROM customers WHERE id=? AND company_id=? AND deleted_at IS NULL LIMIT 1");
      $chk->execute([$customerId, $companyId]);
      if (!$chk->fetch()) {
        throw new \RuntimeException("Customer not found");
      }

      $st = $pdo->prepare("
        INSERT INTO invoices_quotations
          (company_id, customer_id, type, status, doc_no, issue_date, due_date, currency,
           subtotal, total_tax, total_amount, notes, created_by)
        VALUES
          (?,?,?,?,?,?,?,?,?,?,?,?,?)
      ");
      $st->execute([
        $companyId, $customerId, $type, 'draft', $docNo, $issueDate,
        ($dueDate !== '' ? $dueDate : null),
        $currency,
        round($subtotal,2), round($totalTax,2), $totalAmount,
        $notes,
        $userId
      ]);

      $docId = (int)$pdo->lastInsertId();

      $itSt = $pdo->prepare("
        INSERT INTO invoice_items
          (company_id, doc_id, line_no, title, description, quantity, unit, unit_price, vat_rate, vat_amount, line_total)
        VALUES
          (?,?,?,?,?,?,?,?,?,?,?)
      ");

      foreach ($cleanItems as $ci) {
        $itSt->execute([
          $companyId, $docId, $ci['line_no'], $ci['title'], $ci['description'],
          $ci['quantity'], $ci['unit'], $ci['unit_price'], $ci['vat_rate'], $ci['vat_amount'], $ci['line_total']
        ]);
      }

      // Audit log
      $a = $pdo->prepare("
        INSERT INTO audit_logs (company_id,user_id,action,entity,entity_id,ip_address,user_agent)
        VALUES (?,?,?,?,?,?,?)
      ");
      $a->execute([
        $companyId, $userId, 'DOC_CREATED', 'document', $docId,
        Security::ip(),
        substr((string)($_SERVER['HTTP_USER_AGENT'] ?? ''), 0, 255)
      ]);

      $pdo->commit();

      // şimdilik list'e dön
      Auth::redirect('/docs');
    } catch (\Throwable $e) {
      if ($pdo->inTransaction()) $pdo->rollBack();
      http_response_code(500);
      echo "Save error: " . htmlspecialchars($e->getMessage());
    }
  }

  // GET /docs
  public function list(): void
  {
    $pdo = DB::pdo();
    $cid = Tenant::companyId();

    $st = $pdo->prepare("
      SELECT iq.id, iq.type, iq.status, iq.doc_no, iq.issue_date, iq.total_amount, iq.currency,
             c.name AS customer_name
      FROM invoices_quotations iq
      JOIN customers c ON c.id=iq.customer_id AND c.company_id=iq.company_id
      WHERE iq.company_id=? AND iq.deleted_at IS NULL
      ORDER BY iq.id DESC
      LIMIT 200
    ");
    $st->execute([$cid]);
    $docs = $st->fetchAll();

    $title = "Belgeler";

    ob_start();
    require __DIR__ . '/../Views/documents/list_inner.php';
    $content = ob_get_clean();

    require __DIR__ . '/../Views/layouts/main.php';
  }
}
