<?php
declare(strict_types=1);

namespace App\Core;

final class App
{
  public static function run(): void
  {
    self::boot();

    $router = new Router();
    $router->get('/login',  ['App\Controllers\AuthController', 'showLogin']);
    $router->post('/login', ['App\Controllers\AuthController', 'login']);
    $router->post('/logout',['App\Controllers\AuthController', 'logout']);

    $router->get('/',       ['App\Controllers\DashboardController', 'index'], ['auth']);
    $router->get('/docs',   ['App\Controllers\DocumentsController', 'list'],  ['auth']);
    $router->get('/docs/create', ['App\Controllers\DocumentsController','createForm'], ['auth']);
    $router->post('/docs/create',['App\Controllers\DocumentsController','createSave'], ['auth']);
    $router->get('/docs/view',   ['App\Controllers\DocumentsController','view'],      ['auth']);

    $router->post('/docs/delete-request', ['App\Controllers\DeleteRequestsController','request'], ['auth']);
    $router->get('/delete-requests',      ['App\Controllers\DeleteRequestsController','list'],   ['auth','role:admin']);
    $router->post('/delete-requests/decide',['App\Controllers\DeleteRequestsController','decide'],['auth','role:admin']);

    $router->get('/products', ['App\Controllers\ProductsController','list'], ['auth']);
    $router->get('/products/search', ['App\Controllers\ProductsController','search'], ['auth']);
    
    $router->get('/profile',  ['App\Controllers\ProfileController', 'show'],   ['auth']);
    $router->post('/profile', ['App\Controllers\ProfileController', 'update'], ['auth']);


    $router->dispatch();
  }

  private static function boot(): void
  {
    $app = require __DIR__ . '/../../config/app.php';
    session_name($app['session_name']);
    session_start();

    DB::init(require __DIR__ . '/../../config/database.php');

   // Tenant sadece tenant gerektiren sayfalarda çözülsün (dev kolaylığı)
	$uri = (string)parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
	if (!str_contains($uri, '/login')) {
	  Tenant::resolveFromRequest();
	} else {
	  // login sayfası tenant'siz çalışsın (dev)
	  if (empty($_SESSION['tenant_company_id'])) {
		$_SESSION['tenant_company_id'] = 1;
		$_SESSION['tenant_company_slug'] = 'demo';
	  }
	}

    I18n::init($app['default_lang'], $app['supported_langs']);

    // Basit güvenlik header'ları
    header('X-Frame-Options: DENY');
    header('X-Content-Type-Options: nosniff');
    header('Referrer-Policy: strict-origin-when-cross-origin');
  }
}
