<?php
declare(strict_types=1);

namespace App\Core;

final class Router
{
    private array $routes = ['GET' => [], 'POST' => []];

    public function get(string $path, array $handler, array $mw = []): void
    {
        $this->routes['GET'][$path] = [$handler, $mw];
    }

    public function post(string $path, array $handler, array $mw = []): void
    {
        $this->routes['POST'][$path] = [$handler, $mw];
    }

    public function dispatch(): void
    {
        $method = $_SERVER['REQUEST_METHOD'] ?? 'GET';

        // 1. Gerçek URL path'ini al
        $rawPath = (string) parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);

        // 2. Base path'i kırp (örn: /elegance/public)
        $base = \App\Core\Url::basePath();
        if ($base !== '' && str_starts_with($rawPath, $base)) {
            $rawPath = substr($rawPath, strlen($base));
        }

        // 3. Normalize et
        $path = '/' . trim($rawPath, '/');
        if ($path === '/') $path = '/';

        // 4. Route bul
        [$handler, $middlewares] = $this->routes[$method][$path] ?? [null, null];

        if (!$handler) {
            http_response_code(404);
            echo "404 - Route bulunamadı: " . htmlspecialchars($path);
            return;
        }

        // 5. Middleware çalıştır
        $this->runMiddlewares($middlewares ?? []);

        // 6. Controller çağır
        [$class, $action] = $handler;
        (new $class())->{$action}();
    }

    private function runMiddlewares(array $middlewares): void
    {
        foreach ($middlewares as $m) {
            if ($m === 'auth') {
                Auth::requireLogin();
            } elseif (str_starts_with($m, 'role:')) {
                $role = substr($m, 5);
                Auth::requireRole($role);
            }
        }
    }
}
