<?php
declare(strict_types=1);

use App\Core\Csrf;
use App\Core\Security;

$csrf = Csrf::token();
?>
<div class="d-flex flex-column flex-md-row align-items-md-center justify-content-between gap-2">
  <div>
    <h1 class="h5 mb-1">Fatura / Teklif Oluştur</h1>
    <div class="text-secondary small">Kalem ekle, ürün seç, hızlıca kaydet.</div>
  </div>
  <div class="d-flex gap-2">
    <a class="btn btn-sm btn-outline-secondary" href="/elegance/public/docs">Listeye Dön</a>
  </div>
</div>

<hr class="my-3">

<form method="post" action="/elegance/public/docs/create" class="needs-validation" novalidate>
  <input type="hidden" name="csrf" value="<?= Security::e($csrf) ?>">

  <div class="row g-3">
    <div class="col-12 col-md-3">
      <label class="form-label">Tür</label>
      <select name="type" class="form-select">
        <option value="invoice">Fatura</option>
        <option value="quote">Teklif</option>
      </select>
    </div>

    <div class="col-12 col-md-5">
      <label class="form-label">Müşteri</label>
      <select name="customer_id" class="form-select" required>
        <option value="">Seçiniz</option>
        <?php foreach (($customers ?? []) as $c): ?>
          <option value="<?= (int)$c['id'] ?>"><?= Security::e((string)$c['name']) ?></option>
        <?php endforeach; ?>
      </select>
      <div class="invalid-feedback">Lütfen müşteri seçin.</div>
      <div class="form-text">Müşteri yoksa önce customers eklemeliyiz (sonraki adımda ekleyeceğiz).</div>
    </div>

    <div class="col-6 col-md-2">
      <label class="form-label">Tarih</label>
      <input type="date" name="issue_date" class="form-control" value="<?= date('Y-m-d') ?>">
    </div>

    <div class="col-6 col-md-2">
      <label class="form-label">Vade</label>
      <input type="date" name="due_date" class="form-control">
    </div>

    <div class="col-12 col-md-3">
      <label class="form-label">Para Birimi</label>
      <select name="currency" class="form-select">
        <option value="TRY">TRY</option>
        <option value="USD">USD</option>
        <option value="EUR">EUR</option>
      </select>
    </div>
  </div>

  <hr class="my-4">

  <div class="d-flex flex-column flex-md-row justify-content-between align-items-md-center gap-2 mb-2">
    <h2 class="h6 m-0">Kalemler</h2>
    <div class="d-flex gap-2">
      <button type="button" class="btn btn-sm btn-outline-secondary"
              data-bs-toggle="modal" data-bs-target="#productPicker">
        📦 Ürün Seç
      </button>
      <button type="button" class="btn btn-sm btn-outline-primary" id="addRow">+ Satır Ekle</button>
    </div>
  </div>

  <div class="table-responsive">
    <table class="table table-sm align-middle" id="itemsTable">
      <thead>
        <tr>
          <th>Ürün/Hizmet</th>
          <th style="width:140px">Miktar</th>
          <th style="width:160px">Birim Fiyat</th>
          <th style="width:110px">KDV %</th>
          <th style="width:70px"></th>
        </tr>
      </thead>
      <tbody></tbody>
    </table>
  </div>

  <div class="mt-3">
    <label class="form-label">Not</label>
    <textarea class="form-control" name="notes" rows="3" placeholder="Opsiyonel not..."></textarea>
  </div>

  <div class="mt-4 d-flex flex-column flex-sm-row gap-2">
    <button class="btn btn-primary" type="submit">Kaydet</button>
    <a class="btn btn-outline-secondary" href="/elegance/public/docs">Vazgeç</a>
  </div>
</form>

<!-- Product Picker Modal -->
<div class="modal fade" id="productPicker" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content">
      <div class="modal-header">
        <div>
          <div class="fw-semibold">Ürün Seç</div>
          <div class="text-secondary small">Ara, seç ve satıra otomatik ekle.</div>
        </div>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Kapat"></button>
      </div>

      <div class="modal-body">
        <input class="form-control" id="prodSearch" placeholder="Ürün adı veya SKU ara...">
        <div class="mt-3 table-responsive">
          <table class="table table-hover align-middle">
            <thead>
              <tr>
                <th>SKU</th>
                <th>Ürün</th>
                <th class="text-end">Fiyat</th>
                <th class="text-end">KDV</th>
                <th class="text-end"></th>
              </tr>
            </thead>
            <tbody id="prodResults">
              <tr><td colspan="5" class="text-center text-secondary py-3">Arama yap...</td></tr>
            </tbody>
          </table>
        </div>
        <div class="text-secondary small mt-2">
          Not: Arama endpoint’i <code>/elegance/public/products/search</code> olmalı.
        </div>
      </div>
    </div>
  </div>
</div>

<script>
/* Bootstrap form validation (minimal) */
(() => {
  const forms = document.querySelectorAll('.needs-validation');
  Array.from(forms).forEach(form => {
    form.addEventListener('submit', event => {
      if (!form.checkValidity()) {
        event.preventDefault();
        event.stopPropagation();
      }
      form.classList.add('was-validated');
    }, false);
  });
})();

const tbody = document.querySelector('#itemsTable tbody');
const addRowBtn = document.getElementById('addRow');

function escapeHtml(s){
  return String(s ?? '')
    .replaceAll('&','&amp;').replaceAll('<','&lt;')
    .replaceAll('>','&gt;').replaceAll('"','&quot;')
    .replaceAll("'","&#039;");
}

function rowTemplate(prefill = {}) {
  const idx = tbody.children.length;

  const title = prefill.title ?? '';
  const desc  = prefill.description ?? '';
  const unit  = prefill.unit ?? '';
  const qty   = prefill.quantity ?? 1;
  const price = prefill.unit_price ?? 0;
  const vat   = prefill.vat_rate ?? 20;

  return `
    <tr>
      <td>
        <input class="form-control form-control-sm" name="items[${idx}][title]" value="${escapeHtml(title)}"
               placeholder="Örn: Web Tasarım" required>
        <input class="form-control form-control-sm mt-1" name="items[${idx}][description]" value="${escapeHtml(desc)}"
               placeholder="Açıklama (opsiyonel)">
        <input class="form-control form-control-sm mt-1" name="items[${idx}][unit]" value="${escapeHtml(unit)}"
               placeholder="Birim (adet/saat vb.)">
      </td>
      <td><input class="form-control form-control-sm" type="number" step="0.001" min="0"
                 name="items[${idx}][quantity]" value="${qty}"></td>
      <td><input class="form-control form-control-sm" type="number" step="0.01" min="0"
                 name="items[${idx}][unit_price]" value="${price}"></td>
      <td><input class="form-control form-control-sm" type="number" step="0.01" min="0"
                 name="items[${idx}][vat_rate]" value="${vat}"></td>
      <td class="text-end">
        <button type="button" class="btn btn-sm btn-outline-danger delRow">Sil</button>
      </td>
    </tr>
  `;
}

function addRow(prefill = {}) {
  tbody.insertAdjacentHTML('beforeend', rowTemplate(prefill));
}

addRowBtn?.addEventListener('click', () => addRow());

tbody.addEventListener('click', (e) => {
  if (e.target.classList.contains('delRow')) {
    e.target.closest('tr')?.remove();
  }
});

// ilk satır
addRow();

/* Product Search + Select */
const prodSearch  = document.getElementById('prodSearch');
const prodResults = document.getElementById('prodResults');
let timer = null;

async function fetchProducts(q){
  const url = "/elegance/public/products/search?q=" + encodeURIComponent(q);
  const res = await fetch(url, { headers: { 'Accept': 'application/json' } });
  return res.json();
}

function renderProducts(rows){
  if (!rows || rows.length === 0){
    prodResults.innerHTML = `<tr><td colspan="5" class="text-center text-secondary py-3">Sonuç yok.</td></tr>`;
    return;
  }

  prodResults.innerHTML = rows.map(p => {
    const payload = escapeHtml(JSON.stringify(p));
    return `
      <tr>
        <td>${escapeHtml(p.sku || '')}</td>
        <td class="fw-semibold">${escapeHtml(p.name || '')}</td>
        <td class="text-end">${escapeHtml(p.price)}</td>
        <td class="text-end">${escapeHtml(p.vat_rate)}%</td>
        <td class="text-end">
          <button type="button" class="btn btn-sm btn-primary" data-p="${payload}">Seç</button>
        </td>
      </tr>
    `;
  }).join('');
}

prodSearch?.addEventListener('input', () => {
  clearTimeout(timer);
  const q = prodSearch.value.trim();

  timer = setTimeout(async () => {
    if (q.length < 1){
      prodResults.innerHTML = `<tr><td colspan="5" class="text-center text-secondary py-3">Arama yap...</td></tr>`;
      return;
    }
    try{
      const data = await fetchProducts(q);
      renderProducts(data.data || []);
    }catch(e){
      prodResults.innerHTML = `<tr><td colspan="5" class="text-center text-danger py-3">Arama hatası.</td></tr>`;
    }
  }, 250);
});

prodResults?.addEventListener('click', (e) => {
  const btn = e.target.closest('button[data-p]');
  if (!btn) return;

  const p = JSON.parse(btn.getAttribute('data-p'));

  addRow({
    title: p.name,
    unit: p.unit || '',
    unit_price: p.price,
    vat_rate: p.vat_rate,
    quantity: 1
  });

  // modal kapat
  const modalEl = document.getElementById('productPicker');
  const modal = window.bootstrap?.Modal.getInstance(modalEl);
  modal?.hide();
});
</script>
