<?php
declare(strict_types=1);

require __DIR__ . '/inc/db.php';
require __DIR__ . '/inc/auth.php';
require __DIR__ . '/inc/helpers.php';
require __DIR__ . '/inc/csrf.php';

require_login();
$uid = current_user_id();

// Cash accounts (bank/cash/wallet)
$st = $pdo->prepare("SELECT id,name FROM accounts WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$st->execute([$uid]);
$cashAccounts = $st->fetchAll();

// Investment accounts
$st = $pdo->prepare("SELECT * FROM investment_accounts WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$st->execute([$uid]);
$invAccounts = $st->fetchAll();

// Totals per investment account (TRY-based transfers)
$st = $pdo->prepare("
  SELECT ia.id, ia.name,
         COALESCE(SUM(CASE WHEN it.is_active=1 AND it.deleted_at IS NULL AND it.investment_account_id=ia.id AND it.direction='in'  THEN it.amount ELSE 0 END),0) AS total_in,
         COALESCE(SUM(CASE WHEN it.is_active=1 AND it.deleted_at IS NULL AND it.investment_account_id=ia.id AND it.direction='out' THEN it.amount ELSE 0 END),0) AS total_out
  FROM investment_accounts ia
  LEFT JOIN investment_transfers it
    ON it.user_id=ia.user_id
  WHERE ia.user_id=? AND ia.deleted_at IS NULL AND ia.is_active=1
  GROUP BY ia.id, ia.name
  ORDER BY ia.name
");
$st->execute([$uid]);
$invTotals = $st->fetchAll();

$netInvest = 0.0;
foreach ($invTotals as $r) {
  $netInvest += (float)$r['total_in'] - (float)$r['total_out'];
}
?>
<!doctype html>
<html lang="tr" class="h-full">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>Yatırımlar • gelir_gider_v1</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <script>tailwind.config={darkMode:'class'};</script>
  <link rel="stylesheet" href="assets/css/app.css" />
</head>
<body class="h-full bg-slate-50 text-slate-900 dark:bg-slate-950 dark:text-slate-100">
<div class="min-h-screen flex">

  <!-- Sidebar -->
  <aside id="sidebar"
    class="fixed lg:static inset-y-0 left-0 z-50 w-72 -translate-x-full lg:translate-x-0 transition-transform
           bg-white/90 dark:bg-slate-950/90 backdrop-blur border-r border-slate-200 dark:border-slate-800">
    <div class="h-16 px-4 flex items-center justify-between border-b border-slate-200 dark:border-slate-800">
      <div class="flex items-center gap-3">
        <div class="h-10 w-10 rounded-2xl bg-slate-900 dark:bg-white flex items-center justify-center">
          <span class="text-white dark:text-slate-900 font-black">₺</span>
        </div>
        <div>
          <p class="font-semibold leading-5">gelir_gider_v1</p>
          <p class="text-xs muted">Panel</p>
        </div>
      </div>
      <button id="sidebarClose" class="btn lg:hidden" type="button">✕</button>
    </div>

    <nav class="p-3 space-y-1">
      <a class="nav-item" href="./">📊 Dashboard</a>
      <a class="nav-item" href="transactions.php">🧾 İşlemler</a>
      <a class="nav-item" href="accounts.php">🏦 Hesaplar</a>
      <a class="nav-item" href="categories.php">🏷️ Kategoriler</a>
      <a class="nav-item" href="goals.php">🎯 Hedefler</a>
      <a class="nav-item nav-active" href="investments.php">📈 Yatırımlar</a>
      <div class="pt-3 mt-3 border-t border-slate-200 dark:border-slate-800">
        <a class="nav-item" href="logout.php">🚪 Çıkış</a>
      </div>
    </nav>
  </aside>

  <div id="overlay" class="fixed inset-0 bg-black/40 z-40 hidden lg:hidden"></div>

  <div class="flex-1">
    <!-- Topbar -->
    <header class="sticky top-0 z-30 border-b border-slate-200/70 dark:border-slate-800/70 bg-white/70 dark:bg-slate-950/70 backdrop-blur">
      <div class="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 py-3 flex items-center justify-between">
        <div class="flex items-center gap-2">
          <button id="sidebarOpen" class="btn lg:hidden" type="button">☰</button>
          <div>
            <p class="font-semibold leading-5">Yatırımlar</p>
            <p class="text-xs muted">Hesap • Transfer • Özet</p>
          </div>
        </div>

        <div class="flex items-center gap-2">
          <button id="themeToggle" class="btn" type="button" aria-label="Tema Değiştir">
            <span id="themeIcon">🌙</span>
            <span class="hidden sm:inline">Karanlık Mod</span>
          </button>
          <a class="btn" href="./">← Dashboard</a>
        </div>
      </div>
    </header>

    <main class="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 py-6 space-y-6">

      <!-- Summary -->
      <section class="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div class="card p-4 md:col-span-2">
          <p class="text-sm muted">Net Yatırıma Gönderilen (v1)</p>
          <p class="mt-2 text-3xl font-black"><?php echo fmt_try((float)$netInvest); ?></p>
          <p class="mt-1 text-sm muted">Nakit→Yatırım gönder - Yatırımdan çekilen</p>
        </div>
        <div class="card p-4">
          <p class="text-sm muted">Yatırım hesabı</p>
          <p class="mt-2 text-3xl font-black"><?php echo count($invAccounts); ?></p>
          <p class="mt-1 text-sm muted">Aktif yatırım hesapları</p>
        </div>
      </section>

      <!-- Create investment account -->
      <section class="card p-4">
        <div class="flex items-start justify-between gap-3">
          <div>
            <p class="font-semibold">Yatırım Hesabı Oluştur</p>
            <p class="text-sm muted">Borsa / Kripto / Altın / Fon</p>
          </div>
          <a class="btn" href="investments_trades.php">📌 Varlık & İşlemler</a>
        </div>

        <form class="mt-4 grid grid-cols-1 md:grid-cols-12 gap-3" method="post" action="api/investments.php">
          <input type="hidden" name="csrf" value="<?php echo htmlspecialchars(csrf_token()); ?>">
          <input type="hidden" name="action" value="create_account">

          <div class="md:col-span-5">
            <label class="text-sm font-medium">Hesap Adı</label>
            <input name="name" class="sel w-full mt-1" required placeholder="Örn: Binance / Garanti Yatırım / Altın">
          </div>

          <div class="md:col-span-4">
            <label class="text-sm font-medium">Tür</label>
            <select name="type" class="sel w-full mt-1">
              <option value="borsa" class="bg-white dark:bg-slate-900">Borsa</option>
              <option value="kripto" class="bg-white dark:bg-slate-900">Kripto</option>
              <option value="altin" class="bg-white dark:bg-slate-900">Altın</option>
              <option value="fon" class="bg-white dark:bg-slate-900">Fon</option>
            </select>
          </div>

          <div class="md:col-span-3">
            <label class="text-sm font-medium">Para Birimi</label>
            <select name="currency" class="sel w-full mt-1">
              <option value="TRY" class="bg-white dark:bg-slate-900">TRY</option>
              <option value="USD" class="bg-white dark:bg-slate-900">USD</option>
              <option value="EUR" class="bg-white dark:bg-slate-900">EUR</option>
            </select>
          </div>

          <div class="md:col-span-12 flex justify-end">
            <button class="btn" type="submit">✅ Kaydet</button>
          </div>
        </form>
      </section>

      <!-- Transfer money -->
      <section class="card p-4">
        <div class="flex items-start justify-between gap-3">
          <div>
            <p class="font-semibold">Para Transferi</p>
            <p class="text-sm muted">Nakit→Yatırım / Yatırım→Nakit / Yatırım→Yatırım</p>
          </div>
        </div>

        <form class="mt-4 grid grid-cols-1 md:grid-cols-12 gap-3" method="post" action="api/investments.php">
          <input type="hidden" name="csrf" value="<?php echo htmlspecialchars(csrf_token()); ?>">
          <input type="hidden" name="action" value="transfer">

          <div class="md:col-span-4">
            <label class="text-sm font-medium">Transfer Türü</label>
            <select name="transfer_type" id="transferType" class="sel w-full mt-1">
              <option value="cash_to_invest">Nakit Hesap → Yatırım</option>
              <option value="invest_to_cash">Yatırım → Nakit Hesap</option>
              <option value="invest_to_invest">Yatırım → Yatırım</option>
            </select>
          </div>

          <div class="md:col-span-4" id="cashWrap">
            <label class="text-sm font-medium">Nakit Hesap</label>
            <select name="cash_account_id" class="sel w-full mt-1">
              <?php foreach($cashAccounts as $a): ?>
                <option value="<?php echo (int)$a['id']; ?>" class="bg-white dark:bg-slate-900"><?php echo htmlspecialchars((string)$a['name']); ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="md:col-span-4" id="fromInvWrap">
            <label class="text-sm font-medium">Kaynak Yatırım Hesabı</label>
            <select name="from_investment_account_id" class="sel w-full mt-1">
              <?php foreach($invAccounts as $ia): ?>
                <option value="<?php echo (int)$ia['id']; ?>" class="bg-white dark:bg-slate-900"><?php echo htmlspecialchars((string)$ia['name']); ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="md:col-span-4" id="toInvWrap">
            <label class="text-sm font-medium">Hedef Yatırım Hesabı</label>
            <select name="to_investment_account_id" class="sel w-full mt-1">
              <?php foreach($invAccounts as $ia): ?>
                <option value="<?php echo (int)$ia['id']; ?>" class="bg-white dark:bg-slate-900"><?php echo htmlspecialchars((string)$ia['name']); ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="md:col-span-3">
            <label class="text-sm font-medium">Tutar (₺)</label>
            <input name="amount" type="number" step="0.01" min="0" class="sel w-full mt-1" required>
          </div>

          <div class="md:col-span-3">
            <label class="text-sm font-medium">Tarih</label>
            <input name="occurred_at" type="date" class="sel w-full mt-1" value="<?php echo date('Y-m-d'); ?>" required>
          </div>

          <div class="md:col-span-6">
            <label class="text-sm font-medium">Not</label>
            <input name="note" class="sel w-full mt-1" placeholder="Opsiyonel">
          </div>

          <div class="md:col-span-12 flex justify-end">
            <button class="btn" type="submit">✅ Transferi Kaydet</button>
          </div>
        </form>
      </section>

      <!-- Accounts list -->
      <section class="card p-4">
        <div class="flex items-start justify-between">
          <div>
            <p class="font-semibold">Yatırım Hesapları Özeti</p>
            <p class="text-sm muted">Şimdilik para transferlerinden net durumu gösterir (varlık ekranı: “Varlık & İşlemler”)</p>
          </div>
        </div>

        <div class="mt-4 overflow-auto">
          <table class="w-full text-sm">
            <thead class="text-xs muted">
              <tr class="border-b border-slate-200 dark:border-slate-800">
                <th class="py-2 text-left font-medium">Hesap</th>
                <th class="py-2 text-right font-medium">Toplam Giriş</th>
                <th class="py-2 text-right font-medium">Toplam Çıkış</th>
                <th class="py-2 text-right font-medium">Net</th>
              </tr>
            </thead>
            <tbody class="divide-y divide-slate-200 dark:divide-slate-800">
              <?php foreach($invTotals as $r):
                $net = (float)$r['total_in'] - (float)$r['total_out'];
              ?>
              <tr class="hover:bg-slate-50 dark:hover:bg-slate-800/40">
                <td class="py-2 pr-2 font-semibold"><?php echo htmlspecialchars((string)$r['name']); ?></td>
                <td class="py-2 text-right text-emerald-600 dark:text-emerald-400"><?php echo fmt_try((float)$r['total_in']); ?></td>
                <td class="py-2 text-right text-rose-600 dark:text-rose-400"><?php echo fmt_try((float)$r['total_out']); ?></td>
                <td class="py-2 text-right font-bold"><?php echo fmt_try((float)$net); ?></td>
              </tr>
              <?php endforeach; ?>
              <?php if(!$invTotals): ?>
                <tr><td colspan="4" class="py-6 text-center muted">Henüz yatırım hesabı yok.</td></tr>
              <?php endif; ?>
            </tbody>
          </table>
        </div>
      </section>

    </main>
  </div>
</div>

<script>
  // transfer type UI logic
  const tSel = document.getElementById('transferType');
  const cashWrap = document.getElementById('cashWrap');
  const fromInvWrap = document.getElementById('fromInvWrap');
  const toInvWrap = document.getElementById('toInvWrap');

  function syncTransferUI(){
    const t = tSel.value;
    if (t === 'cash_to_invest') {
      cashWrap.classList.remove('hidden');
      fromInvWrap.classList.add('hidden');
      toInvWrap.classList.remove('hidden');
    } else if (t === 'invest_to_cash') {
      cashWrap.classList.remove('hidden');
      fromInvWrap.classList.remove('hidden');
      toInvWrap.classList.add('hidden');
    } else {
      cashWrap.classList.add('hidden');
      fromInvWrap.classList.remove('hidden');
      toInvWrap.classList.remove('hidden');
    }
  }
  tSel.addEventListener('change', syncTransferUI);
  syncTransferUI();
</script>

<script src="assets/js/app.js"></script>
</body>
</html>
