<?php
declare(strict_types=1);

require __DIR__ . '/inc/db.php';
require __DIR__ . '/inc/auth.php';
require __DIR__ . '/inc/helpers.php';
require __DIR__ . '/inc/csrf.php';

require_login();
$uid = current_user_id();

$inv_id = (int)($_GET['investment_account_id'] ?? 0);

// investment accounts
$st = $pdo->prepare("SELECT id,name,type FROM investment_accounts WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$st->execute([$uid]);
$invAccounts = $st->fetchAll();

if ($inv_id <= 0 && $invAccounts) {
  $inv_id = (int)$invAccounts[0]['id'];
}

// selected investment account
$selected = null;
foreach ($invAccounts as $ia) { if ((int)$ia['id'] === $inv_id) { $selected = $ia; break; } }

// trade list
$tr = $pdo->prepare("
  SELECT * FROM investment_trades
  WHERE user_id=? AND investment_account_id=? AND deleted_at IS NULL AND is_active=1
  ORDER BY occurred_at DESC, id DESC
  LIMIT 200
");
$tr->execute([$uid, $inv_id]);
$trades = $tr->fetchAll();

// holdings: buy - sell
$st = $pdo->prepare("
  SELECT asset,
    SUM(CASE WHEN side='buy'  THEN quantity ELSE -quantity END) AS qty
  FROM investment_trades
  WHERE user_id=? AND investment_account_id=? AND deleted_at IS NULL AND is_active=1
  GROUP BY asset
  HAVING qty <> 0
  ORDER BY asset
");
$st->execute([$uid, $inv_id]);
$holdings = $st->fetchAll();

$chartLabels = [];
$chartValues = [];
foreach ($holdings as $h) {
  $chartLabels[] = (string)$h['asset'];
  $chartValues[] = (float)$h['qty'];
}
?>
<!doctype html>
<html lang="tr" class="h-full">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>Varlık & İşlemler • gelir_gider_v1</title>

  <script src="https://cdn.tailwindcss.com"></script>
  <script>tailwind.config={darkMode:'class'};</script>
  <link rel="stylesheet" href="assets/css/app.css" />
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>

<body class="h-full bg-slate-50 text-slate-900 dark:bg-slate-950 dark:text-slate-100">
<div class="min-h-screen flex">

  <!-- Sidebar -->
  <aside id="sidebar"
    class="fixed lg:static inset-y-0 left-0 z-50 w-72 -translate-x-full lg:translate-x-0 transition-transform
           bg-white/90 dark:bg-slate-950/90 backdrop-blur border-r border-slate-200 dark:border-slate-800">
    <div class="h-16 px-4 flex items-center justify-between border-b border-slate-200 dark:border-slate-800">
      <div class="flex items-center gap-3">
        <div class="h-10 w-10 rounded-2xl bg-slate-900 dark:bg-white flex items-center justify-center">
          <span class="text-white dark:text-slate-900 font-black">₺</span>
        </div>
        <div>
          <p class="font-semibold leading-5">gelir_gider_v1</p>
          <p class="text-xs muted">Panel</p>
        </div>
      </div>
      <button id="sidebarClose" class="btn lg:hidden" type="button">✕</button>
    </div>

    <nav class="p-3 space-y-1">
      <a class="nav-item" href="./">📊 Dashboard</a>
      <a class="nav-item" href="transactions.php">🧾 İşlemler</a>
      <a class="nav-item" href="accounts.php">🏦 Hesaplar</a>
      <a class="nav-item" href="categories.php">🏷️ Kategoriler</a>
      <a class="nav-item" href="goals.php">🎯 Hedefler</a>
      <a class="nav-item nav-active" href="investments.php">📈 Yatırımlar</a>
      <div class="pt-3 mt-3 border-t border-slate-200 dark:border-slate-800">
        <a class="nav-item" href="logout.php">🚪 Çıkış</a>
      </div>
    </nav>
  </aside>

  <div id="overlay" class="fixed inset-0 bg-black/40 z-40 hidden lg:hidden"></div>

  <div class="flex-1">
    <!-- Topbar -->
    <header class="sticky top-0 z-30 border-b border-slate-200/70 dark:border-slate-800/70 bg-white/70 dark:bg-slate-950/70 backdrop-blur">
      <div class="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 py-3 flex items-center justify-between">
        <div class="flex items-center gap-2">
          <button id="sidebarOpen" class="btn lg:hidden" type="button">☰</button>
          <div>
            <p class="font-semibold leading-5">Varlık & İşlemler</p>
            <p class="text-xs muted">Al/Sat • Portföy • Grafik</p>
          </div>
        </div>

        <div class="flex items-center gap-2">
          <button id="themeToggle" class="btn" type="button" aria-label="Tema Değiştir">
            <span id="themeIcon">🌙</span>
            <span class="hidden sm:inline">Karanlık Mod</span>
          </button>
          <a class="btn" href="investments.php">← Yatırımlar</a>
        </div>
      </div>
    </header>

    <main class="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 py-6 space-y-6">

      <!-- account selector -->
      <section class="card p-4">
        <form method="get" class="flex flex-col md:flex-row md:items-end gap-3">
          <div class="flex-1">
            <label class="text-sm font-medium">Yatırım Hesabı Seç</label>
            <select name="investment_account_id" class="sel w-full mt-1" onchange="this.form.submit()">
              <?php foreach($invAccounts as $ia): ?>
                <option value="<?php echo (int)$ia['id']; ?>" <?php if((int)$ia['id']===$inv_id) echo 'selected'; ?> class="bg-white dark:bg-slate-900">
                  <?php echo htmlspecialchars((string)$ia['name']); ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="text-sm muted"><?php echo $selected ? 'Tür: '.htmlspecialchars((string)$selected['type']) : ''; ?></div>
        </form>
      </section>

      <!-- holdings + chart -->
      <section class="grid grid-cols-1 lg:grid-cols-2 gap-4">
        <div class="card p-4">
          <p class="font-semibold">Portföy (v1)</p>
          <p class="text-sm muted">Şimdilik adet bazlı gösterim (sonra fiyat bağlarız)</p>

          <div class="mt-4 overflow-auto">
            <table class="w-full text-sm">
              <thead class="text-xs muted">
                <tr class="border-b border-slate-200 dark:border-slate-800">
                  <th class="py-2 text-left font-medium">Varlık</th>
                  <th class="py-2 text-right font-medium">Adet</th>
                </tr>
              </thead>
              <tbody class="divide-y divide-slate-200 dark:divide-slate-800">
                <?php foreach($holdings as $h): ?>
                  <tr class="hover:bg-slate-50 dark:hover:bg-slate-800/40">
                    <td class="py-2 font-semibold"><?php echo htmlspecialchars((string)$h['asset']); ?></td>
                    <td class="py-2 text-right"><?php echo number_format((float)$h['qty'], 8, ',', '.'); ?></td>
                  </tr>
                <?php endforeach; ?>
                <?php if(!$holdings): ?>
                  <tr><td colspan="2" class="py-6 text-center muted">Bu hesapta varlık yok.</td></tr>
                <?php endif; ?>
              </tbody>
            </table>
          </div>
        </div>

        <div class="card p-4">
          <p class="font-semibold">Dağılım Grafiği</p>
          <p class="text-sm muted">Seçilen yatırım hesabındaki varlıklar</p>
          <div class="mt-4">
            <canvas id="donut"></canvas>
          </div>
        </div>
      </section>

      <!-- add trade -->
      <section class="card p-4">
        <div class="flex items-start justify-between gap-3">
          <div>
            <p class="font-semibold">Varlık İşlemi Ekle</p>
            <p class="text-sm muted">Örn: 1 BTC al / 0.5 ETH sat</p>
          </div>
        </div>

        <form class="mt-4 grid grid-cols-1 md:grid-cols-12 gap-3" method="post" action="api/investment_trades.php">
          <input type="hidden" name="csrf" value="<?php echo htmlspecialchars(csrf_token()); ?>">
          <input type="hidden" name="action" value="create">
          <input type="hidden" name="investment_account_id" value="<?php echo (int)$inv_id; ?>">

          <div class="md:col-span-2">
            <label class="text-sm font-medium">Al/Sat</label>
            <select name="side" class="sel w-full mt-1">
              <option value="buy" class="bg-white dark:bg-slate-900">Al</option>
              <option value="sell" class="bg-white dark:bg-slate-900">Sat</option>
            </select>
          </div>

          <div class="md:col-span-3">
            <label class="text-sm font-medium">Varlık</label>
            <input name="asset" class="sel w-full mt-1" required placeholder="BTC / ETH / XAU / XRP">
          </div>

          <div class="md:col-span-3">
            <label class="text-sm font-medium">Adet</label>
            <input name="quantity" type="number" step="0.00000001" min="0" class="sel w-full mt-1" required>
          </div>

          <div class="md:col-span-2">
            <label class="text-sm font-medium">Fiyat</label>
            <input name="price" type="number" step="0.0001" min="0" class="sel w-full mt-1" value="0">
          </div>

          <div class="md:col-span-2">
            <label class="text-sm font-medium">Tarih</label>
            <input name="occurred_at" type="date" class="sel w-full mt-1" value="<?php echo date('Y-m-d'); ?>" required>
          </div>

          <div class="md:col-span-12">
            <label class="text-sm font-medium">Not</label>
            <input name="note" class="sel w-full mt-1" placeholder="Opsiyonel">
          </div>

          <div class="md:col-span-12 flex justify-end">
            <button class="btn" type="submit">✅ Kaydet</button>
          </div>
        </form>
      </section>

      <!-- trade list -->
      <section class="card p-4">
        <div class="flex items-start justify-between">
          <div>
            <p class="font-semibold">İşlem Geçmişi</p>
            <p class="text-sm muted">Son 200 kayıt</p>
          </div>
        </div>

        <div class="mt-4 overflow-auto">
          <table class="w-full text-sm">
            <thead class="text-xs muted">
              <tr class="border-b border-slate-200 dark:border-slate-800">
                <th class="py-2 text-left font-medium">Tarih</th>
                <th class="py-2 text-left font-medium">Varlık</th>
                <th class="py-2 text-left font-medium">İşlem</th>
                <th class="py-2 text-right font-medium">Adet</th>
                <th class="py-2 text-right font-medium">Fiyat</th>
                <th class="py-2 text-right font-medium">Sil</th>
              </tr>
            </thead>
            <tbody class="divide-y divide-slate-200 dark:divide-slate-800">
              <?php foreach($trades as $t): ?>
              <tr class="hover:bg-slate-50 dark:hover:bg-slate-800/40">
                <td class="py-2 muted"><?php echo htmlspecialchars((string)$t['occurred_at']); ?></td>
                <td class="py-2 font-semibold"><?php echo htmlspecialchars((string)$t['asset']); ?></td>
                <td class="py-2">
                  <span class="badge <?php echo $t['side']==='buy'?'badge-pos':'badge-neg'; ?>">
                    <?php echo $t['side']==='buy'?'AL':'SAT'; ?>
                  </span>
                </td>
                <td class="py-2 text-right"><?php echo number_format((float)$t['quantity'], 8, ',', '.'); ?></td>
                <td class="py-2 text-right"><?php echo number_format((float)$t['price'], 4, ',', '.'); ?></td>
                <td class="py-2 text-right">
                  <form method="post" action="api/investment_trades.php" onsubmit="return confirm('Silinsin mi?');">
                    <input type="hidden" name="csrf" value="<?php echo htmlspecialchars(csrf_token()); ?>">
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" value="<?php echo (int)$t['id']; ?>">
                    <input type="hidden" name="investment_account_id" value="<?php echo (int)$inv_id; ?>">
                    <button class="btn" type="submit">🗑️</button>
                  </form>
                </td>
              </tr>
              <?php endforeach; ?>

              <?php if(!$trades): ?>
                <tr><td colspan="6" class="py-6 text-center muted">İşlem yok.</td></tr>
              <?php endif; ?>
            </tbody>
          </table>
        </div>
      </section>

    </main>
  </div>
</div>

<script>
  const labels = <?php echo json_encode($chartLabels, JSON_UNESCAPED_UNICODE); ?>;
  const values = <?php echo json_encode($chartValues, JSON_UNESCAPED_UNICODE); ?>;

  const ctx = document.getElementById('donut');
  if (ctx && labels.length) {
    new Chart(ctx, {
      type: 'doughnut',
      data: { labels, datasets: [{ data: values }] },
      options: {
        plugins: { legend: { position: 'bottom' } }
      }
    });
  }
</script>

<script src="assets/js/app.js"></script>
</body>
</html>
