<?php
declare(strict_types=1);

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/helpers.php';

$uid = current_user_id();
if (!$uid) { json_out(['ok'=>false,'error'=>'unauthorized'], 401); }

function table_exists(PDO $pdo, string $table): bool {
  try {
    $st = $pdo->prepare("SELECT 1 FROM information_schema.tables WHERE table_schema=DATABASE() AND table_name=? LIMIT 1");
    $st->execute([$table]);
    return (bool)$st->fetchColumn();
  } catch (Throwable $e) {
    return false;
  }
}

function range_from_params(string $period, int $year, int $month): array {
  // return [start,end] Y-m-d inclusive
  if ($year > 0 && $month > 0) {
    $start = sprintf('%04d-%02d-01', $year, $month);
    $dt = new DateTime($start);
    $end = $dt->format('Y-m-t');
    return [$start, $end];
  }
  if ($year > 0 && $month === 0) {
    return [sprintf('%04d-01-01', $year), sprintf('%04d-12-31', $year)];
  }
  // fallback to helpers period_range
  return period_range($period);
}

$period = (string)($_GET['period'] ?? 'this_month');
$year = (int)($_GET['year'] ?? 0);
$month = (int)($_GET['month'] ?? 0);
if ($month < 0 || $month > 12) $month = 0;

[$start, $end] = range_from_params($period, $year, $month);

// KPIs
$st = $pdo->prepare("
  SELECT
    COALESCE(SUM(CASE WHEN type='income' THEN amount ELSE 0 END),0) AS income,
    COALESCE(SUM(CASE WHEN type='expense' THEN amount ELSE 0 END),0) AS expense
  FROM transactions
  WHERE user_id=? AND deleted_at IS NULL AND occurred_at BETWEEN ? AND ?
");
$st->execute([$uid, $start, $end]);
$k = $st->fetch() ?: ['income'=>0,'expense'=>0];

$income = (float)$k['income'];
$expense = (float)$k['expense'];
$net = $income - $expense;

// Category breakdown
function cat_breakdown(PDO $pdo, int $uid, string $type, string $start, string $end): array {
  $st = $pdo->prepare("
    SELECT COALESCE(c.name,'(Kategorisiz)') AS name,
           COALESCE(SUM(t.amount),0) AS amount
    FROM transactions t
    LEFT JOIN categories c ON c.id=t.category_id
    WHERE t.user_id=? AND t.deleted_at IS NULL
      AND t.type=? AND t.occurred_at BETWEEN ? AND ?
    GROUP BY name
    ORDER BY amount DESC
  ");
  $st->execute([$uid, $type, $start, $end]);
  $rows = $st->fetchAll() ?: [];
  $total = 0.0;
  foreach ($rows as $r) $total += (float)$r['amount'];
  $out = [];
  foreach ($rows as $r){
    $amt = (float)$r['amount'];
    $pct = $total > 0 ? ($amt / $total) * 100.0 : 0.0;
    $out[] = ['name'=>(string)$r['name'], 'amount'=>$amt, 'pct'=>$pct];
  }
  return $out;
}

$expense_by_cat = cat_breakdown($pdo, $uid, 'expense', $start, $end);
$income_by_cat  = cat_breakdown($pdo, $uid, 'income', $start, $end);

// By account summary
$st = $pdo->prepare("
  SELECT a.id, a.name,
    COALESCE(SUM(CASE WHEN t.type='income' AND t.account_id=a.id THEN t.amount ELSE 0 END),0) AS income,
    COALESCE(SUM(CASE WHEN t.type='expense' AND t.account_id=a.id THEN t.amount ELSE 0 END),0) AS expense,
    COALESCE(SUM(CASE WHEN t.type='transfer' AND t.to_account_id=a.id THEN t.amount ELSE 0 END),0) AS transfer_in,
    COALESCE(SUM(CASE WHEN t.type='transfer' AND t.from_account_id=a.id THEN t.amount ELSE 0 END),0) AS transfer_out
  FROM accounts a
  LEFT JOIN transactions t
    ON t.user_id=a.user_id AND t.deleted_at IS NULL AND t.occurred_at BETWEEN ? AND ?
  WHERE a.user_id=? AND a.deleted_at IS NULL AND a.is_active=1
  GROUP BY a.id, a.name
  ORDER BY a.name
");
$st->execute([$start, $end, $uid]);
$accs = $st->fetchAll() ?: [];
$by_account = [];
foreach ($accs as $a){
  $inc = (float)$a['income'];
  $exp = (float)$a['expense'];
  $tin = (float)$a['transfer_in'];
  $tout = (float)$a['transfer_out'];
  $transfer_net = $tin - $tout;
  $netAcc = $inc - $exp + $transfer_net;
  $by_account[] = [
    'id'=>(int)$a['id'],
    'name'=>(string)$a['name'],
    'income'=>$inc,
    'expense'=>$exp,
    'transfer_net'=>$transfer_net,
    'net'=>$netAcc
  ];
}

// Daily max
function max_day(PDO $pdo, int $uid, string $type, string $start, string $end): array {
  $st = $pdo->prepare("
    SELECT occurred_at AS d, SUM(amount) AS total
    FROM transactions
    WHERE user_id=? AND deleted_at IS NULL AND type=? AND occurred_at BETWEEN ? AND ?
    GROUP BY occurred_at
    ORDER BY total DESC
    LIMIT 1
  ");
  $st->execute([$uid, $type, $start, $end]);
  $r = $st->fetch();
  if (!$r) return ['label'=>'—', 'value'=>0.0];
  return ['label'=>(string)$r['d'].' • ₺'.number_format((float)$r['total'],0,',','.'), 'value'=>(float)$r['total']];
}

$maxExp = max_day($pdo, $uid, 'expense', $start, $end);
$maxInc = max_day($pdo, $uid, 'income', $start, $end);

$days = 1;
try {
  $d1 = new DateTime($start);
  $d2 = new DateTime($end);
  $days = max(1, (int)$d1->diff($d2)->format('%a') + 1);
} catch (Throwable $e) { $days = 1; }

$summary = [
  'avg_daily_expense' => $expense / $days,
  'max_expense_day' => $maxExp['label'],
  'max_income_day' => $maxInc['label'],
];

// Last tx
$st = $pdo->prepare("
  SELECT id, occurred_at, type, amount, description
  FROM transactions
  WHERE user_id=? AND deleted_at IS NULL AND occurred_at BETWEEN ? AND ?
  ORDER BY occurred_at DESC, id DESC
  LIMIT 20
");
$st->execute([$uid, $start, $end]);
$last_tx = [];
foreach (($st->fetchAll() ?: []) as $r){
  $last_tx[] = [
    'id'=>(int)$r['id'],
    'date'=>(string)$r['occurred_at'],
    'type'=>(string)$r['type'],
    'amount'=>(float)$r['amount'],
    'desc'=>(string)($r['description'] ?? ''),
  ];
}

// Cards (optional)
$cards = ['total_debt'=>0.0,'spend'=>0.0,'payments'=>0.0,'cash_advance'=>0.0];
if (table_exists($pdo, 'credit_cards')) {
  try {
    // total debt = SUM(used_amount) if exists
    $st = $pdo->prepare("SELECT
      COALESCE(SUM(used_amount),0) AS debt
      FROM credit_cards
      WHERE user_id=? AND deleted_at IS NULL AND is_active=1");
    $st->execute([$uid]);
    $cards['total_debt'] = (float)($st->fetchColumn() ?: 0);

    // if credit_card_transactions exists with type column
    if (table_exists($pdo, 'credit_card_transactions')) {
      // spend = purchase, payments = payment, cash advance = cash_advance (best effort)
      $st = $pdo->prepare("
        SELECT
          COALESCE(SUM(CASE WHEN type IN ('purchase','spend') THEN amount ELSE 0 END),0) AS spend,
          COALESCE(SUM(CASE WHEN type IN ('payment','pay') THEN amount ELSE 0 END),0) AS payments,
          COALESCE(SUM(CASE WHEN type IN ('cash_advance','cash') THEN amount ELSE 0 END),0) AS cash_adv
        FROM credit_card_transactions
        WHERE user_id=? AND deleted_at IS NULL AND occurred_at BETWEEN ? AND ?
      ");
      $st->execute([$uid, $start, $end]);
      $r = $st->fetch() ?: ['spend'=>0,'payments'=>0,'cash_adv'=>0];
      $cards['spend'] = (float)$r['spend'];
      $cards['payments'] = (float)$r['payments'];
      $cards['cash_advance'] = (float)$r['cash_adv'];
    }
  } catch (Throwable $e) { /* keep zeros */ }
}

// Investments (optional - best effort)
$investments = ['portfolio_est'=>0.0,'period_pl'=>0.0,'asset_count'=>0,'by_asset'=>[]];
if (table_exists($pdo, 'investment_holdings')) {
  try {
    $st = $pdo->prepare("
      SELECT symbol, COALESCE(SUM(quantity),0) AS qty
      FROM investment_holdings
      WHERE user_id=? AND deleted_at IS NULL
      GROUP BY symbol
      ORDER BY symbol
    ");
    $st->execute([$uid]);
    $rows = $st->fetchAll() ?: [];
    foreach ($rows as $r){
      $investments['by_asset'][] = [
        'symbol'=>(string)$r['symbol'],
        'qty'=>(float)$r['qty'],
        'est_try'=>0.0
      ];
    }
    $investments['asset_count'] = count($investments['by_asset']);
  } catch (Throwable $e) {}
}

// Insights (simple)
$insights = [];
if ($expense > 0 && $income > 0) {
  $ratio = $expense / max(1.0, $income);
  if ($ratio > 0.9) $insights[] = "Giderlerin gelire çok yakın. Bu dönemde tasarruf alanı arayabilirsin.";
  elseif ($ratio < 0.5) $insights[] = "Giderlerin gelire göre düşük. Birikim için iyi bir dönem.";
}
if (!empty($expense_by_cat[0]['name'] ?? '')) {
  $top = $expense_by_cat[0];
  $insights[] = "En çok gider: {$top['name']} (₺".number_format((float)$top['amount'],0,',','.').").";
}
if ($maxExp['label'] !== '—') $insights[] = "En yüksek gider günü: {$maxExp['label']}.";
if ($maxInc['label'] !== '—') $insights[] = "En yüksek gelir günü: {$maxInc['label']}.";

json_out([
  'ok'=>true,
  'range'=>['start'=>$start,'end'=>$end],
  'kpis'=>['income'=>$income,'expense'=>$expense,'net'=>$net],
  'expense_by_cat'=>$expense_by_cat,
  'income_by_cat'=>$income_by_cat,
  'by_account'=>$by_account,
  'summary'=>$summary,
  'last_tx'=>$last_tx,
  'cards'=>$cards,
  'investments'=>$investments,
  'insights'=>$insights,
]);
