<?php
declare(strict_types=1);

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/helpers.php';

$uid = current_user_id();
if (!$uid) json_out(['ok'=>false,'error'=>'unauthorized'], 401);

$period = (string)($_GET['period'] ?? 'this_month');
[$start, $end] = period_range($period);

// Normalize to dates (YYYY-MM-DD)
$start = substr($start, 0, 10);
$end   = substr($end, 0, 10);

function table_exists(PDO $pdo, string $table): bool {
  $st = $pdo->prepare("
    SELECT COUNT(*) c
    FROM information_schema.tables
    WHERE table_schema = DATABASE() AND table_name = ?
  ");
  $st->execute([$table]);
  return (int)$st->fetchColumn() > 0;
}

function pct_delta(float $now, float $prev): float {
  if ($prev == 0.0) return $now == 0.0 ? 0.0 : 100.0;
  return (($now - $prev) / abs($prev)) * 100.0;
}

function prev_period_range(string $period): ?array {
  $today = new DateTime('today');
  if ($period === 'this_month') {
    return [(new DateTime('first day of last month'))->format('Y-m-d'), (new DateTime('last day of last month'))->format('Y-m-d')];
  }
  if ($period === 'last_month') {
    return [(new DateTime('first day of -2 month'))->format('Y-m-d'), (new DateTime('last day of -2 month'))->format('Y-m-d')];
  }
  if ($period === 'this_year') {
    $y = (int)$today->format('Y') - 1;
    return [sprintf('%04d-01-01',$y), sprintf('%04d-12-31',$y)];
  }
  return null;
}

function sum_income_expense(PDO $pdo, int $uid, string $start, string $end): array {
  $st = $pdo->prepare("
    SELECT
      COALESCE(SUM(CASE WHEN type='income'  THEN amount ELSE 0 END),0) AS income,
      COALESCE(SUM(CASE WHEN type='expense' THEN amount ELSE 0 END),0) AS expense
    FROM transactions
    WHERE user_id=? AND deleted_at IS NULL
      AND occurred_at BETWEEN ? AND ?
      AND type IN ('income','expense')
  ");
  $st->execute([$uid, $start, $end]);
  $row = $st->fetch() ?: ['income'=>0,'expense'=>0];
  $income = (float)($row['income'] ?? 0);
  $expense = (float)($row['expense'] ?? 0);
  return ['income'=>$income, 'expense'=>$expense, 'net'=>$income-$expense];
}

// ---- KPIs (income/expense/net + deltas)
$sum = sum_income_expense($pdo, $uid, $start, $end);
$income  = (float)$sum['income'];
$expense = (float)$sum['expense'];
$net     = (float)$sum['net'];

$incomeDelta = 0.0; $expenseDelta = 0.0; $netDelta = 0.0;
$prev = prev_period_range($period);
if ($prev) {
  [$ps,$pe] = $prev;
  $prevSum = sum_income_expense($pdo, $uid, $ps, $pe);
  $incomeDelta  = pct_delta($income,  (float)$prevSum['income']);
  $expenseDelta = pct_delta($expense, (float)$prevSum['expense']);
  $netDelta     = pct_delta($net,     (float)$prevSum['net']);
}

// ---- Accounts balances (computed)
$st = $pdo->prepare("
  SELECT a.id, a.name,
    (a.opening_balance
      + COALESCE((
        SELECT SUM(
          CASE
            WHEN t.type='income'  AND t.account_id=a.id THEN t.amount
            WHEN t.type='expense' AND t.account_id=a.id THEN -t.amount
            WHEN t.type='transfer' AND t.to_account_id=a.id THEN t.amount
            WHEN t.type='transfer' AND t.from_account_id=a.id THEN -t.amount
            ELSE 0
          END
        )
        FROM transactions t
        WHERE t.user_id=a.user_id AND t.deleted_at IS NULL
      ),0)
    ) AS balance
  FROM accounts a
  WHERE a.user_id=? AND a.deleted_at IS NULL AND a.is_active=1
  ORDER BY a.name
");
$st->execute([$uid]);
$accRows = $st->fetchAll();

$totalBalance = 0.0;
$accounts = [];
foreach ($accRows as $r) {
  $bal = (float)($r['balance'] ?? 0);
  $totalBalance += $bal;
  $accounts[] = ['name'=>(string)$r['name'], 'balance'=>$bal];
}

// ---- Credit card debt (supports both schemas)
$cardDebt = 0.0;
if (table_exists($pdo, 'credit_cards')) {
  if (table_exists($pdo, 'credit_card_transactions')) {
    $st = $pdo->prepare("
      SELECT COALESCE(SUM(
        COALESCE(c.opening_used,0) +
        COALESCE((
          SELECT SUM(
            CASE
              WHEN x.type IN ('charge','cash_advance') THEN x.amount
              WHEN x.type='payment' THEN -x.amount
              ELSE 0
            END
          )
          FROM credit_card_transactions x
          WHERE x.user_id=c.user_id AND x.credit_card_id=c.id AND x.deleted_at IS NULL
        ),0)
      ),0) AS debt
      FROM credit_cards c
      WHERE c.user_id=? AND c.deleted_at IS NULL AND c.is_active=1
    ");
    $st->execute([$uid]);
    $cardDebt = (float)$st->fetchColumn();
  } else {
    $st = $pdo->prepare("
      SELECT COALESCE(SUM(COALESCE(opening_used,0)),0)
      FROM credit_cards
      WHERE user_id=? AND deleted_at IS NULL AND is_active=1
    ");
    $st->execute([$uid]);
    $cardDebt = (float)$st->fetchColumn();
  }
}

// ---- Portfolio (investment transfers net; keeps it consistent with top KPI)
$portfolio = 0.0;
$portfolioByAccount = [];
if (table_exists($pdo, 'investment_accounts') && table_exists($pdo, 'investment_transfers')) {
  $st = $pdo->prepare("
    SELECT ia.id, ia.name,
           COALESCE(SUM(
             CASE
               WHEN it.direction='in'  THEN it.amount
               WHEN it.direction='out' THEN -it.amount
               ELSE 0
             END
           ),0) AS total
    FROM investment_accounts ia
    LEFT JOIN investment_transfers it
      ON it.investment_account_id=ia.id
     AND it.user_id=ia.user_id
     AND it.deleted_at IS NULL AND it.is_active=1
    WHERE ia.user_id=? AND ia.deleted_at IS NULL AND ia.is_active=1
    GROUP BY ia.id, ia.name
    ORDER BY ia.name
  ");
  $st->execute([$uid]);
  foreach ($st->fetchAll() as $r) {
    $val = (float)($r['total'] ?? 0);
    $portfolio += $val;
    $portfolioByAccount[] = ['name'=>(string)$r['name'], 'value'=>$val];
  }
} elseif (table_exists($pdo, 'investment_transfers')) {
  $st = $pdo->prepare("
    SELECT COALESCE(SUM(
      CASE WHEN direction='in' THEN amount
           WHEN direction='out' THEN -amount
           ELSE 0 END
    ),0)
    FROM investment_transfers
    WHERE user_id=? AND deleted_at IS NULL AND is_active=1
  ");
  $st->execute([$uid]);
  $portfolio = (float)$st->fetchColumn();
}

$portfolioDelta = 0.0; // optional later

// ---- Trend (daily for month; monthly for year)
$trend = [];
$groupByMonth = in_array($period, ['this_year','last_year'], true) || (strlen($start)===10 && substr($start,0,4) !== substr($end,0,4));
if ($groupByMonth) {
  $st = $pdo->prepare("
    SELECT DATE_FORMAT(occurred_at, '%Y-%m-01') AS bucket,
           COALESCE(SUM(CASE WHEN type='income'  THEN amount ELSE 0 END),0) AS income,
           COALESCE(SUM(CASE WHEN type='expense' THEN amount ELSE 0 END),0) AS expense
    FROM transactions
    WHERE user_id=? AND deleted_at IS NULL
      AND occurred_at BETWEEN ? AND ?
      AND type IN ('income','expense')
    GROUP BY bucket
    ORDER BY bucket
  ");
} else {
  $st = $pdo->prepare("
    SELECT DATE(occurred_at) AS bucket,
           COALESCE(SUM(CASE WHEN type='income'  THEN amount ELSE 0 END),0) AS income,
           COALESCE(SUM(CASE WHEN type='expense' THEN amount ELSE 0 END),0) AS expense
    FROM transactions
    WHERE user_id=? AND deleted_at IS NULL
      AND occurred_at BETWEEN ? AND ?
      AND type IN ('income','expense')
    GROUP BY bucket
    ORDER BY bucket
  ");
}
$st->execute([$uid, $start, $end]);
foreach ($st->fetchAll() as $r) {
  $inc = (float)($r['income'] ?? 0);
  $exp = (float)($r['expense'] ?? 0);
  $trend[] = [
    'bucket' => (string)$r['bucket'],
    'income' => $inc,
    'expense' => $exp,
    'net' => $inc - $exp,
  ];
}

// ---- Highlights (Öne Çıkanlar)
$high = [
  'max_expense_bucket' => null,
  'best_net_bucket' => null,
  'top_expense_category' => null,
  'top_income_category' => null,
  'portfolio_pl' => null,
];

try {
  // Bucket comparisons within selected period (month-level)
  $st = $pdo->prepare("
    SELECT DATE_FORMAT(occurred_at,'%Y-%m-01') AS bucket,
           COALESCE(SUM(CASE WHEN type='income'  THEN amount ELSE 0 END),0) AS income,
           COALESCE(SUM(CASE WHEN type='expense' THEN amount ELSE 0 END),0) AS expense
    FROM transactions
    WHERE user_id=? AND deleted_at IS NULL
      AND occurred_at BETWEEN ? AND ?
      AND type IN ('income','expense')
    GROUP BY bucket
  ");
  $st->execute([$uid, $start, $end]);
  $rows = $st->fetchAll();
  if ($rows) {
    $maxExp = null; $bestNet = null;
    foreach ($rows as $r) {
      $bucket = (string)$r['bucket'];
      $inc = (float)$r['income']; $exp = (float)$r['expense'];
      if ($maxExp === null || $exp > $maxExp['value']) $maxExp = ['bucket'=>$bucket,'value'=>$exp];
      $netv = $inc - $exp;
      if ($bestNet === null || $netv > $bestNet['value']) $bestNet = ['bucket'=>$bucket,'value'=>$netv];
    }
    $high['max_expense_bucket'] = $maxExp;
    $high['best_net_bucket'] = $bestNet;
  }

  // Top categories
  if (table_exists($pdo, 'categories')) {
    $st = $pdo->prepare("
      SELECT c.name, SUM(t.amount) AS total
      FROM transactions t
      JOIN categories c ON c.id=t.category_id
      WHERE t.user_id=? AND t.deleted_at IS NULL
        AND t.occurred_at BETWEEN ? AND ?
        AND t.type='expense'
      GROUP BY c.id, c.name
      ORDER BY total DESC
      LIMIT 1
    ");
    $st->execute([$uid, $start, $end]);
    $r = $st->fetch();
    if ($r) $high['top_expense_category'] = ['name'=>(string)$r['name'], 'total'=>(float)$r['total']];

    $st = $pdo->prepare("
      SELECT c.name, SUM(t.amount) AS total
      FROM transactions t
      JOIN categories c ON c.id=t.category_id
      WHERE t.user_id=? AND t.deleted_at IS NULL
        AND t.occurred_at BETWEEN ? AND ?
        AND t.type='income'
      GROUP BY c.id, c.name
      ORDER BY total DESC
      LIMIT 1
    ");
    $st->execute([$uid, $start, $end]);
    $r = $st->fetch();
    if ($r) $high['top_income_category'] = ['name'=>(string)$r['name'], 'total'=>(float)$r['total']];
  }

  // Portfolio P/L: if you have trades table later, compute here. For now keep null so UI can show "—"
  $high['portfolio_pl'] = null;
} catch (Throwable $e) {
  // ignore highlight errors
}

// ---- Goals (top 3 nearest by target_date)
$goals = [];
if (table_exists($pdo, 'goals')) {
  $st = $pdo->prepare("
    SELECT g.id, g.name, g.target_amount, g.target_date,
           COALESCE(SUM(gc.amount),0) AS contributed
    FROM goals g
    LEFT JOIN goal_contributions gc
      ON gc.goal_id=g.id
     AND gc.user_id=g.user_id
     AND gc.deleted_at IS NULL
    WHERE g.user_id=? AND g.deleted_at IS NULL AND g.is_active=1
    GROUP BY g.id, g.name, g.target_amount, g.target_date
    ORDER BY (g.target_date IS NULL) ASC, g.target_date ASC, g.id DESC
    LIMIT 3
  ");
  $st->execute([$uid]);
  foreach ($st->fetchAll() as $r) {
    $target = (float)($r['target_amount'] ?? 0);
    $contrib = (float)($r['contributed'] ?? 0);
    $pct = $target > 0 ? ($contrib / $target) * 100.0 : 0.0;
    $goals[] = [
      'id' => (int)$r['id'],
      'name' => (string)$r['name'],
      'target_amount' => $target,
      'contributed' => $contrib,
      'pct' => $pct,
      'target_date' => $r['target_date'] ? (string)$r['target_date'] : null,
      'remaining' => max(0.0, $target - $contrib),
    ];
  }
}

// ---- Last transactions
$st = $pdo->prepare("
  SELECT type, amount, occurred_at, description
  FROM transactions
  WHERE user_id=? AND deleted_at IS NULL
  ORDER BY occurred_at DESC, id DESC
  LIMIT 10
");
$st->execute([$uid]);
$last = [];
foreach ($st->fetchAll() as $r) {
  $last[] = [
    'type' => (string)$r['type'],
    'amount' => (float)$r['amount'],
    'date' => (string)$r['occurred_at'],
    'desc' => (string)($r['description'] ?? ''),
  ];
}

json_out([
  'ok' => true,
  'kpis' => [
    'income' => $income,
    'expense' => $expense,
    'net' => $net,
    'income_delta' => $incomeDelta,
    'expense_delta' => $expenseDelta,
    'net_delta' => $netDelta,
    'total_balance' => $totalBalance,
    'card_debt' => $cardDebt,
    'portfolio_est' => $portfolio,
    'portfolio_delta' => $portfolioDelta,
  ],
  'trend' => $trend,
  'accounts' => $accounts,
  'last_tx' => $last,
  'highlights' => $high,
  'investments' => [
    'total_portfolio' => $portfolio,
    'by_account' => $portfolioByAccount,
    'daily_pl' => 0.0,
  ],
  'goals' => $goals,
  'insights' => [],
]);
