<?php
declare(strict_types=1);

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/helpers.php';

$uid = current_user_id();
if (!$uid) json_out(['ok'=>false,'error'=>'unauthorized'], 401);

$period = (string)($_GET['period'] ?? 'this_month');
[$start, $end] = period_range($period);

function table_exists(PDO $pdo, string $table): bool {
  $st = $pdo->prepare("
    SELECT COUNT(*) c
    FROM information_schema.tables
    WHERE table_schema = DATABASE() AND table_name = ?
  ");
  $st->execute([$table]);
  return (int)$st->fetchColumn() > 0;
}

function sum_income_expense(PDO $pdo, int $uid, string $start, string $end): array {
  $st = $pdo->prepare("
    SELECT
      COALESCE(SUM(CASE WHEN type='income'  THEN amount ELSE 0 END),0) AS income,
      COALESCE(SUM(CASE WHEN type='expense' THEN amount ELSE 0 END),0) AS expense
    FROM transactions
    WHERE user_id=? AND deleted_at IS NULL
      AND occurred_at BETWEEN ? AND ?
      AND type IN ('income','expense')
  ");
  $st->execute([$uid, $start, $end]);
  $row = $st->fetch() ?: ['income'=>0,'expense'=>0];
  $income = (float)($row['income'] ?? 0);
  $expense = (float)($row['expense'] ?? 0);
  return ['income'=>$income, 'expense'=>$expense, 'net'=>$income-$expense];
}

function prev_period_range(string $period): ?array {
  $today = new DateTime('today');
  if ($period === 'this_month') {
    return [(new DateTime('first day of last month'))->format('Y-m-d'), (new DateTime('last day of last month'))->format('Y-m-d')];
  }
  if ($period === 'last_month') {
    return [(new DateTime('first day of -2 month'))->format('Y-m-d'), (new DateTime('last day of -2 month'))->format('Y-m-d')];
  }
  if ($period === 'this_year') {
    $y = (int)$today->format('Y') - 1;
    return [sprintf('%04d-01-01',$y), sprintf('%04d-12-31',$y)];
  }
  return null;
}

function pct_delta(float $now, float $prev): float {
  if ($prev == 0.0) return $now == 0.0 ? 0.0 : 100.0;
  return (($now - $prev) / abs($prev)) * 100.0;
}

// KPI income/expense/net
$sum = sum_income_expense($pdo, $uid, $start, $end);
$income  = $sum['income'];
$expense = $sum['expense'];
$net     = $sum['net'];

$prev = prev_period_range($period);
$incomeDelta = 0.0; $expenseDelta = 0.0; $netDelta = 0.0;
if ($prev) {
  [$ps,$pe] = $prev;
  $prevSum = sum_income_expense($pdo, $uid, $ps, $pe);
  $incomeDelta  = pct_delta($income,  (float)$prevSum['income']);
  $expenseDelta = pct_delta($expense, (float)$prevSum['expense']);
  $netDelta     = pct_delta($net,     (float)$prevSum['net']);
}

// total_balance (accounts)
$st = $pdo->prepare("
  SELECT a.id, a.name,
    (a.opening_balance
      + COALESCE((
        SELECT SUM(
          CASE
            WHEN t.type='income'  AND t.account_id=a.id THEN t.amount
            WHEN t.type='expense' AND t.account_id=a.id THEN -t.amount
            WHEN t.type='transfer' AND t.to_account_id=a.id THEN t.amount
            WHEN t.type='transfer' AND t.from_account_id=a.id THEN -t.amount
            ELSE 0
          END
        )
        FROM transactions t
        WHERE t.user_id=a.user_id AND t.deleted_at IS NULL
      ),0)
    ) AS balance
  FROM accounts a
  WHERE a.user_id=? AND a.deleted_at IS NULL AND a.is_active=1
  ORDER BY a.name
");
$st->execute([$uid]);
$accRows = $st->fetchAll();

$totalBalance = 0.0;
$accounts = [];
foreach ($accRows as $r) {
  $bal = (float)($r['balance'] ?? 0);
  $totalBalance += $bal;
  $accounts[] = ['name'=>(string)$r['name'], 'balance'=>$bal];
}

// card_debt
$cardDebt = 0.0;
if (table_exists($pdo, 'credit_cards')) {
  if (table_exists($pdo, 'credit_card_transactions')) {
    $st = $pdo->prepare("
      SELECT
        COALESCE(SUM(
          COALESCE(c.opening_used,0) +
          COALESCE((
            SELECT SUM(
              CASE
                WHEN x.type IN ('charge','cash_advance') THEN x.amount
                WHEN x.type='payment' THEN -x.amount
                ELSE 0
              END
            )
            FROM credit_card_transactions x
            WHERE x.user_id=c.user_id AND x.credit_card_id=c.id AND x.deleted_at IS NULL
          ),0)
        ),0) AS debt
      FROM credit_cards c
      WHERE c.user_id=? AND c.deleted_at IS NULL AND c.is_active=1
    ");
    $st->execute([$uid]);
    $cardDebt = (float)$st->fetchColumn();
  } else {
    $st = $pdo->prepare("
      SELECT COALESCE(SUM(COALESCE(opening_used,0)),0)
      FROM credit_cards
      WHERE user_id=? AND deleted_at IS NULL AND is_active=1
    ");
    $st->execute([$uid]);
    $cardDebt = (float)$st->fetchColumn();
  }
}

// portfolio_est (net transfer)
$portfolio = 0.0;
if (table_exists($pdo, 'investment_transfers')) {
  $st = $pdo->prepare("
    SELECT COALESCE(SUM(
      CASE WHEN direction='in' THEN amount
           WHEN direction='out' THEN -amount
           ELSE 0 END
    ),0)
    FROM investment_transfers
    WHERE user_id=? AND deleted_at IS NULL AND is_active=1
  ");
  $st->execute([$uid]);
  $portfolio = (float)$st->fetchColumn();
}
$portfolioDelta = 0.0;

// investments block for dashboard cards
$investments = [
  'total' => $portfolio,
  'daily_pl' => 0.0,
  'alloc' => []
];

// ---- trend (son 24 ay) ----
$trend = [];
$st = $pdo->prepare("
  SELECT DATE_FORMAT(occurred_at, '%Y-%m') AS m,
         COALESCE(SUM(CASE WHEN type='income' THEN amount ELSE 0 END),0)  AS income,
         COALESCE(SUM(CASE WHEN type='expense' THEN amount ELSE 0 END),0) AS expense
  FROM transactions
  WHERE user_id=? AND deleted_at IS NULL
    AND type IN ('income','expense')
  GROUP BY DATE_FORMAT(occurred_at, '%Y-%m')
  ORDER BY m ASC
");
$st->execute([$uid]);
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

$map = [];
foreach ($rows as $r) {
  $m = (string)$r['m'];
  $inc = (float)($r['income'] ?? 0);
  $exp = (float)($r['expense'] ?? 0);
  $map[$m] = ['m'=>$m,'income'=>$inc,'expense'=>$exp,'net'=>$inc-$exp];
}

$cur = new DateTime('first day of this month');
$cur->modify('-23 months');
for ($i=0; $i<24; $i++){
  $key = $cur->format('Y-m');
  $trend[] = $map[$key] ?? ['m'=>$key,'income'=>0,'expense'=>0,'net'=>0];
  $cur->modify('+1 month');
}

// last_tx (son 10)
$st = $pdo->prepare("
  SELECT type, amount, occurred_at, description
  FROM transactions
  WHERE user_id=? AND deleted_at IS NULL
  ORDER BY occurred_at DESC, id DESC
  LIMIT 10
");
$st->execute([$uid]);
$last = [];
foreach ($st->fetchAll() as $r) {
  $last[] = [
    'type' => (string)$r['type'],
    'amount' => (float)$r['amount'],
    'date' => (string)$r['occurred_at'],
    'desc' => (string)($r['description'] ?? ''),
  ];
}

// goals (top 3)
$goalsOut = [];
if (table_exists($pdo, 'goals')) {
  $st = $pdo->prepare("
    SELECT g.id, g.name, g.target_amount, g.target_date, g.is_completed,
      COALESCE((
        SELECT SUM(gc.amount)
        FROM goal_contributions gc
        WHERE gc.user_id=g.user_id AND gc.goal_id=g.id AND gc.deleted_at IS NULL
      ),0) AS saved_amount
    FROM goals g
    WHERE g.user_id=? AND g.deleted_at IS NULL AND g.is_active=1
    ORDER BY g.is_completed ASC,
             g.target_date IS NULL,
             g.target_date ASC,
             g.id DESC
    LIMIT 3
  ");
  $st->execute([$uid]);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);

  foreach ($rows as $g) {
    $target = (float)($g['target_amount'] ?? 0);
    $saved  = (float)($g['saved_amount'] ?? 0);
    $pct = ($target > 0) ? min(100.0, ($saved / $target) * 100.0) : 0.0;
    $done = ((int)($g['is_completed'] ?? 0) === 1) || ($target > 0 && $saved >= $target);
    $left = max(0.0, $target - $saved);

    $goalsOut[] = [
      'id' => (int)$g['id'],
      'name' => (string)$g['name'],
      'target_amount' => $target,
      'saved_amount' => $saved,
      'pct' => $pct,
      'done' => $done,
      'target_date' => $g['target_date'] ? (string)$g['target_date'] : '',
      'saved_text' => fmt_try($saved),
      'target_text' => fmt_try($target),
      'left_text' => fmt_try($left),
    ];
  }
}

// ---- highlights (öne çıkanlar) ----
$highlights = [
  'worstExpenseMonth' => '—',
  'bestProfitMonth' => '—',
  'topExpenseCategory' => '—',
  'topIncomeCategory' => '—',
  'portfolioPL' => '—',
];

if (!empty($trend)) {
  $worst = null; $best = null;
  foreach ($trend as $t) {
    $m = (string)$t['m'];
    $exp = (float)($t['expense'] ?? 0);
    $netv = (float)($t['net'] ?? 0);
    if ($worst === null || $exp > $worst['expense']) $worst = ['m'=>$m,'expense'=>$exp];
    if ($best === null || $netv > $best['net']) $best = ['m'=>$m,'net'=>$netv];
  }
  if ($worst) $highlights['worstExpenseMonth'] = $worst['m'];
  if ($best)  $highlights['bestProfitMonth']  = $best['m'];
}

try {
  $st = $pdo->prepare("
    SELECT c.name, SUM(t.amount) total
    FROM transactions t
    LEFT JOIN categories c ON c.id=t.category_id
    WHERE t.user_id=? AND t.deleted_at IS NULL
      AND t.type='expense'
      AND t.occurred_at BETWEEN ? AND ?
    GROUP BY c.name
    ORDER BY total DESC
    LIMIT 1
  ");
  $st->execute([$uid, $start, $end]);
  $r = $st->fetch(PDO::FETCH_ASSOC);
  if ($r && !empty($r['name'])) $highlights['topExpenseCategory'] = (string)$r['name'];
} catch (Throwable $e) {}

try {
  $st = $pdo->prepare("
    SELECT c.name, SUM(t.amount) total
    FROM transactions t
    LEFT JOIN categories c ON c.id=t.category_id
    WHERE t.user_id=? AND t.deleted_at IS NULL
      AND t.type='income'
      AND t.occurred_at BETWEEN ? AND ?
    GROUP BY c.name
    ORDER BY total DESC
    LIMIT 1
  ");
  $st->execute([$uid, $start, $end]);
  $r = $st->fetch(PDO::FETCH_ASSOC);
  if ($r && !empty($r['name'])) $highlights['topIncomeCategory'] = (string)$r['name'];
} catch (Throwable $e) {}

$highlights['portfolioPL'] = fmt_try((float)($investments['daily_pl'] ?? 0));

// ---- insights (akıllı içgörüler) ----
$insights = [];

try {
  $st = $pdo->prepare("
    SELECT COUNT(*) FROM transactions
    WHERE user_id=? AND deleted_at IS NULL
      AND occurred_at BETWEEN ? AND ?
      AND type IN ('income','expense','transfer')
  ");
  $st->execute([$uid, $start, $end]);
  $cnt = (int)$st->fetchColumn();
  if ($cnt === 0) $insights[] = "Bu dönemde henüz işlem yok. İlk gelir/gider kaydını ekleyebilirsin.";
} catch (Throwable $e) {}

if ($expense > $income && ($income + $expense) > 0) {
  $insights[] = "Seçilen dönemde giderlerin gelirlerinden yüksek. Harcama kategorilerini gözden geçir.";
}

if ($cardDebt > 0) {
  $insights[] = "Kredi kartı borcun var: " . fmt_try($cardDebt) . ". Ödeme planı yapmayı düşünebilirsin.";
}

if ($net > 0 && $income > 0) {
  $rate = ($net / $income) * 100.0;
  if ($rate >= 20) $insights[] = "Güzel gidiyor: Gelirinin yaklaşık %" . (string)round($rate) . " kadarını biriktirmişsin.";
}

if (!empty($goalsOut)) {
  foreach ($goalsOut as $g) {
    if (!empty($g['target_date']) && empty($g['done'])) {
      $d = DateTime::createFromFormat('Y-m-d', (string)$g['target_date']);
      if ($d) {
        $days = (int)(($d->getTimestamp() - (new DateTime('today'))->getTimestamp()) / 86400);
        if ($days >= 0 && $days <= 14) {
          $insights[] = "Hedefin yaklaşıyor: '" . (string)$g['name'] . "' için " . $days . " gün kaldı.";
        }
      }
    }
  }
}

$insights = array_values(array_slice($insights, 0, 6));

json_out([
  'ok' => true,
  'kpis' => [
    'income' => $income,
    'expense' => $expense,
    'net' => $net,
    'income_delta' => $incomeDelta,
    'expense_delta' => $expenseDelta,
    'net_delta' => $netDelta,
    'total_balance' => $totalBalance,
    'card_debt' => $cardDebt,
    'portfolio_est' => $portfolio,
    'portfolio_delta' => $portfolioDelta,
  ],
  'trend' => $trend,
  'accounts' => $accounts,
  'last_tx' => $last,
  'highlights' => $highlights,
  'investments' => $investments,
  'goals' => $goalsOut,
  'insights' => $insights,
]);
