/* gelir_gider_v1 - Dashboard data renderer
 * Depends on: assets/js/app.js for theme toggle + sidebar.
 */

(function(){
  const $ = (sel) => document.querySelector(sel);
  const el = {
    periodSelect: $('#periodSelect'),
    btn6: $('#btn6'),
    btn12: $('#btn12'),
    txFilter: $('#txFilter'),
    // KPIs
    kpiIncome: $('#kpiIncome'),
    kpiIncomeDelta: $('#kpiIncomeDelta'),
    kpiExpense: $('#kpiExpense'),
    kpiExpenseDelta: $('#kpiExpenseDelta'),
    kpiNet: $('#kpiNet'),
    kpiNetDelta: $('#kpiNetDelta'),
    kpiBalance: $('#kpiBalance'),
    kpiCardDebt: $('#kpiCardDebt'),
    kpiPortfolio: $('#kpiPortfolio'),
    kpiPortfolioDelta: $('#kpiPortfolioDelta'),
    // sections
    trendBars: $('#trendBars'),
    balanceDist: $('#balanceDist'),
    txTable: $('#txTable'),
    insightsList: $('#insightsList'),
    // highlights
    worstExpenseMonth: $('#worstExpenseMonth'),
    bestProfitMonth: $('#bestProfitMonth'),
    topExpenseCategory: $('#topExpenseCategory'),
    topIncomeCategory: $('#topIncomeCategory'),
    portfolioPL: $('#portfolioPL'),
    // investments
    invTotal: $('#invTotal'),
    invDailyPL: $('#invDailyPL'),
    invAlloc: $('#invAlloc'),
  };

  let state = {
    period: (el.periodSelect && el.periodSelect.value) ? el.periodSelect.value : 'this_month',
    bars: 12,
    txType: 'all',
    data: null,
  };

  // ---------- helpers ----------
  const fmtTry = (n) => {
    const v = Number(n || 0);
    // Turkish formatting without Intl deps
    const s = v.toFixed(0).replace(/\B(?=(\d{3})+(?!\d))/g, '.');
    return '₺' + s;
  };

  const pctBadge = (node, pct, mode='neu') => {
    if (!node) return;
    const p = Number(pct || 0);
    node.textContent = (p > 0 ? '+' : '') + Math.round(p) + '%';
    node.classList.remove('badge-pos','badge-neg','badge-neu');
    const cls = mode === 'pos' ? 'badge-pos' : (mode === 'neg' ? 'badge-neg' : 'badge-neu');
    node.classList.add('badge', cls);
  };

  const safeText = (node, text) => { if (node) node.textContent = (text ?? '—'); };

  const apiUrl = (period) => `api/dashboard.php?period=${encodeURIComponent(period || 'this_month')}`;

  async function fetchDashboard(){
    const res = await fetch(apiUrl(state.period), { headers: { 'Accept': 'application/json' } });
    if (!res.ok) throw new Error('API error: ' + res.status);
    const j = await res.json();
    if (!j || j.ok !== true) throw new Error('Bad payload');
    return j;
  }

  // ---------- render ----------
  function renderKpis(kpis){
    kpis = kpis || {};
    if (el.kpiIncome) el.kpiIncome.textContent = fmtTry(kpis.income);
    pctBadge(el.kpiIncomeDelta, kpis.income_delta, 'pos');

    if (el.kpiExpense) el.kpiExpense.textContent = fmtTry(kpis.expense);
    pctBadge(el.kpiExpenseDelta, kpis.expense_delta, 'neg');

    if (el.kpiNet) el.kpiNet.textContent = fmtTry(kpis.net);
    pctBadge(el.kpiNetDelta, kpis.net_delta, 'neu');

    if (el.kpiBalance) el.kpiBalance.textContent = fmtTry(kpis.total_balance);

    if (el.kpiCardDebt) el.kpiCardDebt.textContent = fmtTry(kpis.card_debt);

    if (el.kpiPortfolio) el.kpiPortfolio.textContent = fmtTry(kpis.portfolio_est);
    pctBadge(el.kpiPortfolioDelta, kpis.portfolio_delta, 'neu');
  }

  function renderTrend(trend){
    if (!el.trendBars) return;
    trend = Array.isArray(trend) ? trend : [];
    const slice = trend.slice(-state.bars);

    // Ensure horizontal layout
    el.trendBars.classList.add('grid');
    el.trendBars.style.gridTemplateColumns = `repeat(${Math.max(1, slice.length)}, minmax(0, 1fr))`;
    el.trendBars.style.gridAutoFlow = 'column';

    const max = Math.max(1, ...slice.flatMap(x => [Number(x.income||0), Number(x.expense||0), Math.abs(Number(x.net||0))]));
    el.trendBars.innerHTML = slice.map((m) => {
      const inc = Number(m.income||0);
      const exp = Number(m.expense||0);
      const net = Number(m.net||0);
      const incH = Math.round((inc / max) * 100);
      const expH = Math.round((exp / max) * 100);
      const netH = Math.round((Math.abs(net) / max) * 100);
      const label = String(m.m || '').slice(0,7);
      return `
        <div class="flex flex-col justify-end gap-1">
          <div class="flex items-end gap-1 h-28">
            <div class="w-2 rounded-t bg-emerald-500/80" style="height:${incH}%"></div>
            <div class="w-2 rounded-t bg-rose-500/80" style="height:${expH}%"></div>
            <div class="w-2 rounded-t bg-slate-400/70 dark:bg-slate-500/70" style="height:${netH}%"></div>
          </div>
          <div class="text-[11px] text-slate-500 dark:text-slate-400 text-center">${label || '—'}</div>
        </div>
      `;
    }).join('');
  }

  function renderBalances(accounts){
    if (!el.balanceDist) return;
    accounts = Array.isArray(accounts) ? accounts : [];
    const total = accounts.reduce((s,a)=>s + Number(a.balance||0), 0) || 1;

    el.balanceDist.innerHTML = accounts.map(a=>{
      const bal = Number(a.balance||0);
      const pct = Math.max(0, Math.min(100, (bal/total)*100));
      return `
        <div class="mini-box">
          <div class="flex items-center justify-between">
            <div class="text-sm font-semibold">${escapeHtml(a.name || '')}</div>
            <div class="text-sm font-bold">${fmtTry(bal)}</div>
          </div>
          <div class="mt-2 h-2 rounded-full bg-slate-200 dark:bg-slate-800 overflow-hidden">
            <div class="h-full bg-slate-900 dark:bg-white/80" style="width:${pct}%"></div>
          </div>
        </div>
      `;
    }).join('') || `<div class="muted">Hesap yok.</div>`;
  }

  function renderLastTx(last_tx){
    if (!el.txTable) return;
    last_tx = Array.isArray(last_tx) ? last_tx : [];
    const rows = last_tx
      .filter(r => state.txType === 'all' ? true : (String(r.type||'') === state.txType))
      .slice(0, 10)
      .map(r=>{
        const t = String(r.type||'');
        const label = t==='income'?'Gelir':(t==='expense'?'Gider':'Transfer');
        const badge = t==='income'?'badge-pos':(t==='expense'?'badge-neg':'badge-neu');
        const amt = Number(r.amount||0);
        const sign = t==='expense' ? '-' : '+';
        const amtCls = t==='expense' ? 'text-rose-600 dark:text-rose-400' : 'text-emerald-600 dark:text-emerald-400';
        return `
          <tr class="hover:bg-slate-50 dark:hover:bg-slate-800/40">
            <td class="py-2 pr-2 whitespace-nowrap text-slate-500 dark:text-slate-400">${escapeHtml(r.date || '')}</td>
            <td class="py-2 pr-2"><span class="badge ${badge}">${label}</span></td>
            <td class="py-2 pr-2">
              <div class="font-medium">${escapeHtml(r.desc || '')}</div>
            </td>
            <td class="py-2 text-right font-semibold ${amtCls}">${sign}${fmtTry(amt)}</td>
          </tr>
        `;
      }).join('');
    el.txTable.innerHTML = rows || `<tr><td colspan="4" class="py-6 text-center text-slate-500 dark:text-slate-400">Kayıt yok.</td></tr>`;
  }

  function renderHighlights(h){
    h = h || {};
    safeText(el.worstExpenseMonth, h.worstExpenseMonth || '—');
    safeText(el.bestProfitMonth, h.bestProfitMonth || '—');
    safeText(el.topExpenseCategory, h.topExpenseCategory || '—');
    safeText(el.topIncomeCategory, h.topIncomeCategory || '—');
    safeText(el.portfolioPL, h.portfolioPL || '—');
  }

  function renderInvestments(inv){
    inv = inv || {};
    if (el.invTotal) el.invTotal.textContent = fmtTry(inv.total || 0);
    if (el.invDailyPL) el.invDailyPL.textContent = fmtTry(inv.daily_pl || 0);

    if (!el.invAlloc) return;
    const alloc = Array.isArray(inv.alloc) ? inv.alloc : [];
    el.invAlloc.innerHTML = alloc.map(x=>{
      return `
        <div class="mini-box flex items-center justify-between">
          <div class="text-sm font-semibold">${escapeHtml(x.label || '')}</div>
          <div class="text-sm font-bold">${escapeHtml(x.value || '')}</div>
        </div>
      `;
    }).join('') || `<div class="muted">Henüz yatırım yok.</div>`;
  }

  function renderInsights(list){
    if (!el.insightsList) return;
    list = Array.isArray(list) ? list : [];
    el.insightsList.innerHTML = list.map(t=>`<li class="mini-box">${escapeHtml(String(t))}</li>`).join('')
      || `<li class="muted">Henüz içgörü yok.</li>`;
  }

  function escapeHtml(s){
    return String(s ?? '').replace(/[&<>"']/g, (c)=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[c]));
  }

  function renderAll(payload){
    state.data = payload;
    renderKpis(payload.kpis);
    renderTrend(payload.trend);
    renderBalances(payload.accounts);
    renderLastTx(payload.last_tx);
    renderHighlights(payload.highlights);
    renderInvestments(payload.investments);
    renderInsights(payload.insights);
  }

  async function load(){
    try{
      const j = await fetchDashboard();
      renderAll(j);
    }catch(e){
      console.error(e);
      // Show safe fallbacks
      renderAll({kpis:{}, trend:[], accounts:[], last_tx:[], highlights:{}, investments:{}, insights:["Dashboard verisi alınamadı. api/dashboard.php çalışıyor mu?"]});
    }
  }

  // ---------- events ----------
  if (el.periodSelect){
    el.periodSelect.addEventListener('change', ()=>{
      state.period = el.periodSelect.value || 'this_month';
      load();
    });
  }
  if (el.btn6) el.btn6.addEventListener('click', ()=>{ state.bars = 6; if(state.data) renderTrend(state.data.trend); });
  if (el.btn12) el.btn12.addEventListener('click', ()=>{ state.bars = 12; if(state.data) renderTrend(state.data.trend); });
  if (el.txFilter) el.txFilter.addEventListener('change', ()=>{ state.txType = el.txFilter.value || 'all'; if(state.data) renderLastTx(state.data.last_tx); });

  // initial
  load();
})();
