<?php
declare(strict_types=1);

function backup_dir(): string {
  $dir = dirname(__DIR__) . DIRECTORY_SEPARATOR . 'backups';
  if (!is_dir($dir)) @mkdir($dir, 0777, true);
  return $dir;
}

function backup_list(int $limit=50): array {
  $dir = backup_dir();
  $files = glob($dir . DIRECTORY_SEPARATOR . "*.sql") ?: [];
  rsort($files);
  $out = [];
  foreach (array_slice($files, 0, $limit) as $f) {
    $out[] = [
      'name' => basename($f),
      'size' => filesize($f) ?: 0,
      'mtime' => filemtime($f) ?: time(),
      'path' => $f,
    ];
  }
  return $out;
}

function backup_create(PDO $pdo): array {
  try {
    $db = (string)$pdo->query("SELECT DATABASE()")->fetchColumn();
    if ($db === '') return ['ok'=>false, 'error'=>'DB adı alınamadı (SELECT DATABASE()).'];

    $dir = backup_dir();
    $fname = 'backup_' . date('Y-m-d_His') . '_' . $db . '.sql';
    $path = $dir . DIRECTORY_SEPARATOR . $fname;

    $tables = $pdo->query("SHOW TABLES")->fetchAll(PDO::FETCH_NUM);
    $sql = "-- Kaşmir Finance Backup\n-- DB: {$db}\n-- Created: ".date('c')."\n\nSET NAMES utf8mb4;\nSET FOREIGN_KEY_CHECKS=0;\n\n";

    foreach ($tables as $trow) {
      $table = (string)$trow[0];
      if ($table === '') continue;

      $row = $pdo->query("SHOW CREATE TABLE `{$table}`")->fetch(PDO::FETCH_ASSOC);
      $create = $row['Create Table'] ?? '';
      $sql .= "\n-- ----------------------------\n-- Table: {$table}\n-- ----------------------------\n";
      $sql .= "DROP TABLE IF EXISTS `{$table}`;\n";
      $sql .= $create . ";\n\n";

      $data = $pdo->query("SELECT * FROM `{$table}`")->fetchAll(PDO::FETCH_ASSOC);
      if (!$data) continue;

      foreach ($data as $r) {
        $cols = array_keys($r);
        $vals = [];
        foreach ($cols as $c) {
          $v = $r[$c];
          if ($v === null) $vals[] = "NULL";
          else {
            $vals[] = $pdo->quote((string)$v);
          }
        }
        $sql .= "INSERT INTO `{$table}` (`".implode("`,`",$cols)."`) VALUES (".implode(",",$vals).");\n";
      }
      $sql .= "\n";
    }

    $sql .= "SET FOREIGN_KEY_CHECKS=1;\n";
    file_put_contents($path, $sql);

    return ['ok'=>true, 'file'=>$fname, 'path'=>$path];
  } catch (Throwable $e) {
    return ['ok'=>false, 'error'=>$e->getMessage()];
  }
}
