<?php
declare(strict_types=1);

/**
 * Telegram + SMTP helper
 * - telegram_send(): curl ile Telegram Bot API
 * - smtp_send(): AUTH LOGIN destekli basit SMTP client (SSL/TLS/none)
 */

function telegram_send(string $botToken, string $chatId, string $text): array {
  if ($botToken === '' || $chatId === '') {
    return ['ok'=>false, 'error'=>'Telegram bot token veya chat id boş.'];
  }

  $url = "https://api.telegram.org/bot{$botToken}/sendMessage";
  $payload = http_build_query([
    'chat_id' => $chatId,
    'text' => $text,
    'parse_mode' => 'HTML',
    'disable_web_page_preview' => true
  ]);

  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => $payload,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 15,
  ]);
  $resp = curl_exec($ch);
  $err  = curl_error($ch);
  $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
  curl_close($ch);

  if ($resp === false) return ['ok'=>false, 'error'=>"cURL hata: $err"];
  $json = json_decode($resp, true);
  if ($code >= 200 && $code < 300 && is_array($json) && ($json['ok'] ?? false)) {
    return ['ok'=>true];
  }
  return ['ok'=>false, 'error'=>"Telegram API cevap: ".$resp];
}

function smtp_send(array $cfg, string $toEmail, string $subject, string $bodyHtml): array {
  $host = (string)($cfg['host'] ?? '');
  $port = (int)($cfg['port'] ?? 0);
  $user = (string)($cfg['username'] ?? '');
  $pass = (string)($cfg['password'] ?? '');
  $sec  = (string)($cfg['security'] ?? 'ssl'); // ssl|tls|none
  $from = (string)($cfg['from_email'] ?? $user);
  $fromName = (string)($cfg['from_name'] ?? 'Kaşmir Finance');

  if ($host === '' || $toEmail === '') return ['ok'=>false, 'error'=>'SMTP host veya alıcı e-posta boş.'];
  if ($port <= 0) $port = ($sec === 'ssl') ? 465 : 587;

  $remote = $host . ':' . $port;
  $transport = ($sec === 'ssl') ? "ssl://$remote" : $remote;

  $fp = @stream_socket_client($transport, $errno, $errstr, 15, STREAM_CLIENT_CONNECT);
  if (!$fp) return ['ok'=>false, 'error'=>"SMTP bağlantı hatası: $errstr ($errno)"];

  stream_set_timeout($fp, 15);

  $read = function() use ($fp): string {
    $data = '';
    while (($line = fgets($fp, 515)) !== false) {
      $data .= $line;
      if (preg_match('/^\d{3}\s/', $line)) break;
    }
    return $data;
  };

  $send = function(string $cmd) use ($fp): void {
    fwrite($fp, $cmd . "\r\n");
  };

  $expect = function(array $codes, string $msg) use ($read): array {
    $resp = $read();
    $ok = false;
    foreach ($codes as $c) {
      if (str_starts_with($resp, (string)$c)) { $ok = true; break; }
    }
    return $ok ? ['ok'=>true, 'resp'=>$resp] : ['ok'=>false, 'resp'=>$resp, 'error'=>$msg];
  };

  // Banner
  $r = $expect([220], 'SMTP banner alınamadı'); if (!$r['ok']) { fclose($fp); return $r; }

  $send('EHLO kasmir.local');
  $r = $expect([250], 'EHLO başarısız'); if (!$r['ok']) { fclose($fp); return $r; }

  if ($sec === 'tls') {
    $send('STARTTLS');
    $r = $expect([220], 'STARTTLS başarısız'); if (!$r['ok']) { fclose($fp); return $r; }
    if (!stream_socket_enable_crypto($fp, true, STREAM_CRYPTO_METHOD_TLS_CLIENT)) {
      fclose($fp);
      return ['ok'=>false,'error'=>'TLS crypto etkinleştirilemedi.'];
    }
    $send('EHLO kasmir.local');
    $r = $expect([250], 'EHLO (TLS sonrası) başarısız'); if (!$r['ok']) { fclose($fp); return $r; }
  }

  if ($user !== '' && $pass !== '') {
    $send('AUTH LOGIN');
    $r = $expect([334], 'AUTH LOGIN başlatılamadı'); if (!$r['ok']) { fclose($fp); return $r; }

    $send(base64_encode($user));
    $r = $expect([334], 'SMTP kullanıcı adı kabul edilmedi'); if (!$r['ok']) { fclose($fp); return $r; }

    $send(base64_encode($pass));
    $r = $expect([235], 'SMTP şifre doğrulanamadı'); if (!$r['ok']) { fclose($fp); return $r; }
  }

  $send("MAIL FROM:<$from>");
  $r = $expect([250], 'MAIL FROM başarısız'); if (!$r['ok']) { fclose($fp); return $r; }

  $send("RCPT TO:<$toEmail>");
  $r = $expect([250,251], 'RCPT TO başarısız'); if (!$r['ok']) { fclose($fp); return $r; }

  $send("DATA");
  $r = $expect([354], 'DATA başlatılamadı'); if (!$r['ok']) { fclose($fp); return $r; }

  $subjectEnc = '=?UTF-8?B?' . base64_encode($subject) . '?=';
  $headers = [];
  $headers[] = "From: " . mb_encode_mimeheader($fromName, "UTF-8") . " <$from>";
  $headers[] = "To: <$toEmail>";
  $headers[] = "Subject: $subjectEnc";
  $headers[] = "MIME-Version: 1.0";
  $headers[] = "Content-Type: text/html; charset=UTF-8";

  $data = implode("\r\n", $headers) . "\r\n\r\n" . $bodyHtml . "\r\n.";
  $send($data);

  $r = $expect([250], 'Mesaj gönderilemedi'); if (!$r['ok']) { fclose($fp); return $r; }

  $send('QUIT');
  fclose($fp);
  return ['ok'=>true];
}
