<?php
declare(strict_types=1);

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/functions.php';

$uid = current_user_id() ?? 1;

$err = '';
$ok  = '';

/* -------------------------------------------------------
   Helpers
------------------------------------------------------- */
function sym(string $cur): string {
  return $cur === 'TRY' ? '₺' : ($cur === 'USD' ? '$' : ($cur === 'EUR' ? '€' : $cur.' '));
}

function post_str(string $k): string {
  return trim((string)($_POST[$k] ?? ''));
}
function post_float(string $k): float {
  return (float)($_POST[$k] ?? 0);
}
function post_int(string $k): int {
  return (int)($_POST[$k] ?? 0);
}

/* -------------------------------------------------------
   Actions: create / trash
------------------------------------------------------- */
$action = (string)($_POST['action'] ?? '');

if ($action === 'add_account') {
  $name = post_str('acc_name');
  $typeId = post_int('acc_type_id') ?: null;
  $cur = post_str('acc_currency') ?: 'TRY';
  $opening = post_float('acc_opening_balance');
  $iban = post_str('acc_iban') ?: null;
  $branchName = post_str('acc_branch_name') ?: null;
  $branchCode = post_str('acc_branch_code') ?: null;
  $accNo = post_str('acc_account_no') ?: null;

  if ($name === '') $err = 'Hesap adı zorunlu.';
  if (!$err) {
    $st = $pdo->prepare("
      INSERT INTO accounts
        (user_id, name, account_type_id, currency, opening_balance, iban, branch_name, branch_code, account_no)
      VALUES
        (?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    $st->execute([$uid, $name, $typeId, $cur, $opening, $iban, $branchName, $branchCode, $accNo]);
    $ok = 'Hesap eklendi ✅';
  }
}

if ($action === 'add_card') {
  $name = post_str('card_name');
  $cur = post_str('card_currency') ?: 'TRY';
  $limit = post_float('card_limit');
  $openingDebt = post_float('card_opening_debt');
  $statementDay = post_int('card_statement_day') ?: null;
  $dueDay = post_int('card_due_day') ?: null;

  if ($name === '') $err = 'Kart adı zorunlu.';
  if (!$err) {
    $st = $pdo->prepare("
      INSERT INTO credit_cards
        (user_id, name, currency, card_limit, opening_debt, statement_day, due_day)
      VALUES
        (?, ?, ?, ?, ?, ?, ?)
    ");
    $st->execute([$uid, $name, $cur, $limit, $openingDebt, $statementDay, $dueDay]);
    $ok = 'Kredi kartı eklendi ✅';
  }
}

if ($action === 'add_investment') {
  $name = post_str('inv_name');
  $typeId = post_int('inv_type_id') ?: null;
  $cur = post_str('inv_currency') ?: 'TRY';

  if ($name === '') $err = 'Yatırım hesabı adı zorunlu.';
  if (!$err) {
    $st = $pdo->prepare("
      INSERT INTO investment_accounts
        (user_id, name, investment_type_id, currency)
      VALUES
        (?, ?, ?, ?)
    ");
    $st->execute([$uid, $name, $typeId, $cur]);
    $ok = 'Yatırım hesabı eklendi ✅';
  }
}

// Soft-delete (trash)
if ($action === 'trash_account') {
  $id = post_int('id');
  if ($id > 0) {
    $st = $pdo->prepare("UPDATE accounts SET deleted_at=NOW() WHERE id=? AND user_id=? AND deleted_at IS NULL");
    $st->execute([$id, $uid]);
    $ok = 'Hesap çöp kutusuna taşındı 🗑️';
  }
}
if ($action === 'trash_card') {
  $id = post_int('id');
  if ($id > 0) {
    $st = $pdo->prepare("UPDATE credit_cards SET deleted_at=NOW() WHERE id=? AND user_id=? AND deleted_at IS NULL");
    $st->execute([$id, $uid]);
    $ok = 'Kart çöp kutusuna taşındı 🗑️';
  }
}
if ($action === 'trash_investment') {
  $id = post_int('id');
  if ($id > 0) {
    $st = $pdo->prepare("UPDATE investment_accounts SET deleted_at=NOW() WHERE id=? AND user_id=? AND deleted_at IS NULL");
    $st->execute([$id, $uid]);
    $ok = 'Yatırım hesabı çöp kutusuna taşındı 🗑️';
  }
}

/* -------------------------------------------------------
   Load dropdown categories (account_type, investment_type)
------------------------------------------------------- */
$accTypes = $pdo->prepare("
  SELECT id,name FROM categories
  WHERE user_id=? AND type='account_type' AND deleted_at IS NULL AND is_active=1
  ORDER BY sort_order, name
");
$accTypes->execute([$uid]);
$accTypes = $accTypes->fetchAll();

$invTypes = $pdo->prepare("
  SELECT id,name FROM categories
  WHERE user_id=? AND type='investment_type' AND deleted_at IS NULL AND is_active=1
  ORDER BY sort_order, name
");
$invTypes->execute([$uid]);
$invTypes = $invTypes->fetchAll();

/* -------------------------------------------------------
   Load lists
------------------------------------------------------- */
$accounts = $pdo->prepare("
  SELECT a.*, ct.name AS type_name
  FROM accounts a
  LEFT JOIN categories ct ON ct.id=a.account_type_id
  WHERE a.user_id=? AND a.deleted_at IS NULL AND a.is_active=1
  ORDER BY a.name
");
$accounts->execute([$uid]);
$accounts = $accounts->fetchAll();

$cards = $pdo->prepare("
  SELECT *
  FROM credit_cards
  WHERE user_id=? AND deleted_at IS NULL AND is_active=1
  ORDER BY name
");
$cards->execute([$uid]);
$cards = $cards->fetchAll();

$investments = $pdo->prepare("
  SELECT ia.*, it.name AS type_name
  FROM investment_accounts ia
  LEFT JOIN categories it ON it.id=ia.investment_type_id
  WHERE ia.user_id=? AND ia.deleted_at IS NULL AND ia.is_active=1
  ORDER BY ia.name
");
$investments->execute([$uid]);
$investments = $investments->fetchAll();

/* -------------------------------------------------------
   Summary calculations (MVP)
   - Account balances: opening_balance + net effect from transactions
   - Card debt: opening_debt + expense transactions from credit_card
   - Monthly net: income - expense in current month
------------------------------------------------------- */

// Account movement net per account (income to account, expense from account, transfers in/out)
$accMov = $pdo->prepare("
  SELECT a.id,
    COALESCE(SUM(
      CASE
        WHEN t.deleted_at IS NULL AND t.kind='income' AND t.dest_type='account' AND t.dest_id=a.id THEN t.amount
        WHEN t.deleted_at IS NULL AND t.kind='expense' AND t.source_type='account' AND t.source_id=a.id THEN -t.amount
        WHEN t.deleted_at IS NULL AND t.kind='transfer' AND t.source_type='account' AND t.source_id=a.id THEN -t.amount
        WHEN t.deleted_at IS NULL AND t.kind='transfer' AND t.dest_type='account' AND t.dest_id=a.id THEN t.amount
        ELSE 0
      END
    ),0) AS movement
  FROM accounts a
  LEFT JOIN transactions t ON t.user_id=a.user_id
  WHERE a.user_id=? AND a.deleted_at IS NULL AND a.is_active=1
  GROUP BY a.id
");
$accMov->execute([$uid]);
$accMovRows = $accMov->fetchAll();
$movementByAcc = [];
foreach ($accMovRows as $r) $movementByAcc[(int)$r['id']] = (float)$r['movement'];

$totalBalance = 0.0;
foreach ($accounts as $a) {
  $id = (int)$a['id'];
  $bal = (float)$a['opening_balance'] + ($movementByAcc[$id] ?? 0.0);
  $totalBalance += $bal;
}

// Card debt
$cardSpent = $pdo->prepare("
  SELECT c.id,
    COALESCE(SUM(CASE
      WHEN t.deleted_at IS NULL AND t.kind='expense' AND t.source_type='credit_card' AND t.source_id=c.id THEN t.amount
      ELSE 0
    END),0) AS spent
  FROM credit_cards c
  LEFT JOIN transactions t ON t.user_id=c.user_id
  WHERE c.user_id=? AND c.deleted_at IS NULL AND c.is_active=1
  GROUP BY c.id
");
$cardSpent->execute([$uid]);
$cardSpentRows = $cardSpent->fetchAll();
$spentByCard = [];
foreach ($cardSpentRows as $r) $spentByCard[(int)$r['id']] = (float)$r['spent'];

$totalCardDebt = 0.0;
$totalCardLimit = 0.0;
foreach ($cards as $c) {
  $id = (int)$c['id'];
  $debt = (float)$c['opening_debt'] + ($spentByCard[$id] ?? 0.0);
  $totalCardDebt += $debt;
  $totalCardLimit += (float)$c['card_limit'];
}

// Monthly net (current month)
$monthStart = date('Y-m-01');
$monthEnd   = date('Y-m-t');

$monthSum = $pdo->prepare("
  SELECT
    COALESCE(SUM(CASE WHEN kind='income'  AND deleted_at IS NULL THEN amount ELSE 0 END),0) AS income_sum,
    COALESCE(SUM(CASE WHEN kind='expense' AND deleted_at IS NULL THEN amount ELSE 0 END),0) AS expense_sum
  FROM transactions
  WHERE user_id=? AND occurred_on BETWEEN ? AND ?
");
$monthSum->execute([$uid, $monthStart, $monthEnd]);
$ms = $monthSum->fetch() ?: ['income_sum'=>0,'expense_sum'=>0];
$monthIncome = (float)$ms['income_sum'];
$monthExpense = (float)$ms['expense_sum'];
$monthNet = $monthIncome - $monthExpense;

require __DIR__ . '/../inc/header.php';
?>

<!-- ÜST GENİŞ ÖZET -->
<div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
  <div class="flex flex-col md:flex-row md:items-center justify-between gap-3">
    <div>
      <h2 class="text-lg font-semibold">🏦 Hesaplar</h2>
      <p class="text-sm opacity-70">Hesaplar, kartlar ve yatırım hesapları</p>
    </div>
    <a href="<?= h(page_url('trash.php')) ?>?entity=accounts"
       class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
      🗑️ Çöp Kutusu
    </a>
  </div>

  <?php if ($err): ?>
    <div class="mt-3 p-3 rounded-xl bg-rose-50 dark:bg-rose-950/40 border border-rose-200 dark:border-rose-900 text-sm">
      <?= h($err) ?>
    </div>
  <?php endif; ?>

  <?php if ($ok): ?>
    <div class="mt-3 p-3 rounded-xl bg-emerald-50 dark:bg-emerald-950/40 border border-emerald-200 dark:border-emerald-900 text-sm">
      <?= h($ok) ?>
    </div>
  <?php endif; ?>

  <div class="mt-4 grid grid-cols-1 md:grid-cols-3 gap-4">
    <div class="p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200/60 dark:border-slate-800/60">
      <div class="text-sm opacity-70">Toplam Bakiye</div>
      <div class="text-2xl font-bold mt-1"><?= money($totalBalance) ?></div>
      <div class="text-xs opacity-60 mt-1">Hesap bakiyeleri (açılış + işlemler)</div>
    </div>

    <div class="p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200/60 dark:border-slate-800/60">
      <div class="text-sm opacity-70">Kart Borcu</div>
      <div class="text-2xl font-bold mt-1"><?= money($totalCardDebt) ?></div>
      <div class="text-xs opacity-60 mt-1">
        <?= count($cards) ?> kart • Limit: <?= money($totalCardLimit) ?>
      </div>
    </div>

    <div class="p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200/60 dark:border-slate-800/60">
      <div class="text-sm opacity-70">Bu Ay Net</div>
      <div class="text-2xl font-bold mt-1 text-primary"><?= money($monthNet) ?></div>
      <div class="text-xs opacity-60 mt-1">
        Gelir: <?= money($monthIncome) ?> • Gider: <?= money($monthExpense) ?>
      </div>
    </div>
  </div>
</div>

<!-- TAB MENÜ -->
<div class="mt-6">
  <div class="flex flex-wrap gap-2">
    <button data-tab="tab-acc" class="tabBtn px-4 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 font-medium">
      🏦 Banka / Kasa Hesabı
    </button>
    <button data-tab="tab-card" class="tabBtn px-4 py-2 rounded-xl bg-white/60 dark:bg-slate-950/60 border border-slate-200 dark:border-slate-800">
      💳 Kredi Kartı
    </button>
    <button data-tab="tab-inv" class="tabBtn px-4 py-2 rounded-xl bg-white/60 dark:bg-slate-950/60 border border-slate-200 dark:border-slate-800">
      📈 Yatırımlar
    </button>
  </div>

  <!-- TAB: HESAP -->
  <section id="tab-acc" class="tabPanel mt-4">
    <div class="grid grid-cols-1 xl:grid-cols-2 gap-4">

      <!-- Form -->
      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
        <h3 class="font-semibold text-lg">Banka / Kasa Hesabı Ekle</h3>
        <p class="text-sm opacity-70 mt-1">Açılış bakiyesi pozitif veya negatif olabilir.</p>

        <form class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-3" method="post">
          <input type="hidden" name="action" value="add_account">

          <div class="md:col-span-2">
            <label class="text-sm opacity-70">Hesap Adı</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   name="acc_name" placeholder="Örn: Enpara, Nakit" required>
          </div>

          <div>
            <label class="text-sm opacity-70">Hesap Türü</label>
            <select class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" name="acc_type_id">
              <option value="">Seç...</option>
              <?php foreach ($accTypes as $t): ?>
                <option value="<?= (int)$t['id'] ?>"><?= h($t['name']) ?></option>
              <?php endforeach; ?>
            </select>
            <p class="text-xs opacity-60 mt-1">Bu liste Kategoriler ekranından yönetilecek (account_type).</p>
          </div>

          <div>
            <label class="text-sm opacity-70">Para Birimi</label>
            <select class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" name="acc_currency">
              <option value="TRY">TRY (₺)</option>
              <option value="USD">USD ($)</option>
              <option value="EUR">EUR (€)</option>
            </select>
          </div>

          <div>
            <label class="text-sm opacity-70">Açılış Bakiyesi</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   name="acc_opening_balance" type="number" step="0.01" placeholder="Örn: 10000">
          </div>

          <div>
            <label class="text-sm opacity-70">IBAN (ops.)</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   name="acc_iban" placeholder="TR..">
          </div>

          <div>
            <label class="text-sm opacity-70">Şube Adı (ops.)</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   name="acc_branch_name" placeholder="Örn: Maltepe">
          </div>

          <div>
            <label class="text-sm opacity-70">Şube Kodu (ops.)</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   name="acc_branch_code" placeholder="Örn: 123">
          </div>

          <div>
            <label class="text-sm opacity-70">Hesap No (ops.)</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   name="acc_account_no" placeholder="...">
          </div>

          <div class="md:col-span-2 flex gap-2 pt-2">
            <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">
              ✅ Kaydet
            </button>
            <a href="<?= h(page_url('categories.php')) ?>"
               class="px-4 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
              Türleri Yönet
            </a>
          </div>
        </form>
      </div>

      <!-- Liste -->
      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
        <h3 class="font-semibold text-lg">Hesaplar</h3>
        <p class="text-sm opacity-70 mt-1">Açılış + işlemler neti ile hesaplanan bakiye</p>

        <div class="mt-4 space-y-3">
          <?php if (!$accounts): ?>
            <div class="py-10 text-center opacity-70">Henüz hesap yok.</div>
          <?php endif; ?>

          <?php foreach ($accounts as $a): ?>
            <?php
              $id = (int)$a['id'];
              $bal = (float)$a['opening_balance'] + ($movementByAcc[$id] ?? 0.0);
              $cur = (string)$a['currency'];
            ?>
            <div class="p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 flex items-center justify-between gap-3">
              <div>
                <div class="font-semibold"><?= h($a['name']) ?></div>
                <div class="text-xs opacity-70">
                  <?= h($a['type_name'] ?? '—') ?> • <?= h($cur) ?>
                </div>
              </div>

              <div class="text-right">
                <div class="font-bold whitespace-nowrap"><?= sym($cur) . number_format($bal, 2, ',', '.') ?></div>
                <form method="post" class="mt-2" onsubmit="return confirm('Hesap çöp kutusuna taşınsın mı?');">
                  <input type="hidden" name="action" value="trash_account">
                  <input type="hidden" name="id" value="<?= $id ?>">
                  <button class="px-3 py-1 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
                    🗑️
                  </button>
                </form>
              </div>
            </div>
          <?php endforeach; ?>
        </div>
      </div>

    </div>
  </section>

  <!-- TAB: KREDİ KARTI -->
  <section id="tab-card" class="tabPanel mt-4 hidden">
    <div class="grid grid-cols-1 xl:grid-cols-2 gap-4">

      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
        <h3 class="font-semibold text-lg">Kredi Kartı Ekle</h3>

        <form class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-3" method="post">
          <input type="hidden" name="action" value="add_card">

          <div class="md:col-span-2">
            <label class="text-sm opacity-70">Kart Adı</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
              name="card_name" placeholder="Örn: Garanti Bonus" required>
          </div>

          <div>
            <label class="text-sm opacity-70">Para Birimi</label>
            <select class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" name="card_currency">
              <option value="TRY">TRY (₺)</option>
              <option value="USD">USD ($)</option>
              <option value="EUR">EUR (€)</option>
            </select>
          </div>

          <div>
            <label class="text-sm opacity-70">Limit</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
              name="card_limit" type="number" step="0.01" placeholder="50000">
          </div>

          <div>
            <label class="text-sm opacity-70">Başlangıç Borcu</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
              name="card_opening_debt" type="number" step="0.01" placeholder="0">
          </div>

          <div>
            <label class="text-sm opacity-70">Hesap Kesim Günü</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
              name="card_statement_day" type="number" min="1" max="31" placeholder="5">
          </div>

          <div>
            <label class="text-sm opacity-70">Son Ödeme Günü</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
              name="card_due_day" type="number" min="1" max="31" placeholder="15">
          </div>

          <div class="md:col-span-2 pt-2">
            <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">
              ✅ Kaydet
            </button>
          </div>
        </form>
      </div>

      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
        <h3 class="font-semibold text-lg">Kartlar</h3>
        <p class="text-sm opacity-70 mt-1">Borç = opening_debt + karttan yapılan giderler</p>

        <div class="mt-4 space-y-3">
          <?php if (!$cards): ?>
            <div class="py-10 text-center opacity-70">Henüz kart yok.</div>
          <?php endif; ?>

          <?php foreach ($cards as $c): ?>
            <?php
              $id = (int)$c['id'];
              $cur = (string)$c['currency'];
              $debt = (float)$c['opening_debt'] + ($spentByCard[$id] ?? 0.0);
            ?>
            <div class="p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 flex items-center justify-between gap-3">
              <div>
                <div class="font-semibold"><?= h($c['name']) ?></div>
                <div class="text-xs opacity-70">
                  Limit: <?= sym($cur).number_format((float)$c['card_limit'],2,',','.') ?> • Para: <?= h($cur) ?>
                </div>
              </div>

              <div class="text-right">
                <div class="font-bold whitespace-nowrap">Borç: <?= sym($cur).number_format($debt,2,',','.') ?></div>
                <form method="post" class="mt-2" onsubmit="return confirm('Kart çöp kutusuna taşınsın mı?');">
                  <input type="hidden" name="action" value="trash_card">
                  <input type="hidden" name="id" value="<?= $id ?>">
                  <button class="px-3 py-1 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
                    🗑️
                  </button>
                </form>
              </div>
            </div>
          <?php endforeach; ?>
        </div>

      </div>

    </div>
  </section>

  <!-- TAB: YATIRIM -->
  <section id="tab-inv" class="tabPanel mt-4 hidden">
    <div class="grid grid-cols-1 xl:grid-cols-2 gap-4">

      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
        <h3 class="font-semibold text-lg">Yatırım Hesabı Ekle</h3>

        <form class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-3" method="post">
          <input type="hidden" name="action" value="add_investment">

          <div class="md:col-span-2">
            <label class="text-sm opacity-70">Hesap Adı</label>
            <input class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
              name="inv_name" placeholder="Örn: Binance" required>
          </div>

          <div>
            <label class="text-sm opacity-70">Tür</label>
            <select class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" name="inv_type_id">
              <option value="">Seç...</option>
              <?php foreach ($invTypes as $t): ?>
                <option value="<?= (int)$t['id'] ?>"><?= h($t['name']) ?></option>
              <?php endforeach; ?>
            </select>
            <p class="text-xs opacity-60 mt-1">Bu liste Kategoriler ekranından yönetilecek (investment_type).</p>
          </div>

          <div>
            <label class="text-sm opacity-70">Para Birimi</label>
            <select class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" name="inv_currency">
              <option value="TRY">TRY (₺)</option>
              <option value="USD">USD ($)</option>
              <option value="EUR">EUR (€)</option>
            </select>
          </div>

          <div class="md:col-span-2 pt-2 flex gap-2">
            <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">
              ✅ Kaydet
            </button>
            <a href="<?= h(page_url('categories.php')) ?>"
               class="px-4 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
              Türleri Yönet
            </a>
          </div>
        </form>
      </div>

      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
        <h3 class="font-semibold text-lg">Yatırım Hesapları</h3>

        <div class="mt-4 space-y-3">
          <?php if (!$investments): ?>
            <div class="py-10 text-center opacity-70">Henüz yatırım hesabı yok.</div>
          <?php endif; ?>

          <?php foreach ($investments as $ia): ?>
            <div class="p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 flex items-center justify-between gap-3">
              <div>
                <div class="font-semibold"><?= h($ia['name']) ?></div>
                <div class="text-xs opacity-70"><?= h($ia['type_name'] ?? '—') ?> • <?= h($ia['currency']) ?></div>
              </div>

              <form method="post" onsubmit="return confirm('Yatırım hesabı çöp kutusuna taşınsın mı?');">
                <input type="hidden" name="action" value="trash_investment">
                <input type="hidden" name="id" value="<?= (int)$ia['id'] ?>">
                <button class="px-3 py-1 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
                  🗑️
                </button>
              </form>
            </div>
          <?php endforeach; ?>
        </div>

      </div>
    </div>
  </section>
</div>

<script>
  // Tabs
  const btns = document.querySelectorAll('.tabBtn');
  const panels = document.querySelectorAll('.tabPanel');

  btns.forEach(b=>{
    b.addEventListener('click', ()=>{
      const id = b.dataset.tab;
      panels.forEach(p=>p.classList.add('hidden'));
      document.getElementById(id).classList.remove('hidden');

      btns.forEach(x=>{
        x.classList.remove('font-medium');
        x.classList.add('bg-white/60','dark:bg-slate-950/60');
      });
      b.classList.add('font-medium');
      b.classList.remove('bg-white/60','dark:bg-slate-950/60');
    });
  });
</script>

<?php require __DIR__ . '/../inc/footer.php'; ?>
