<?php
declare(strict_types=1);

$type = 'dashboard'; // header.php aktif menü vs. için

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/functions.php';

$uid = function_exists('require_user_id') ? require_user_id($pdo) : (current_user_id() ?? 0);
if ($uid <= 0) {
  flash_set('error', 'Users tablosunda kullanıcı bulunamadı. Lütfen 1 kullanıcı ekle.');
  header('Location: ' . page_url('index.php'));
  exit;
}

function qstr(string $k, string $def=''): string { return trim((string)($_GET[$k] ?? $def)); }
function qint(string $k, int $def=0): int { return (int)($_GET[$k] ?? $def); }
function date_ok(string $d): bool { return (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $d); }
function money_try(float $v): string { return '₺' . number_format($v, 2, ',', '.'); }

$today = date('Y-m-d');
$from = qstr('from', date('Y-m-01'));
$to   = qstr('to', $today);
if (!date_ok($from)) $from = date('Y-m-01');
if (!date_ok($to))   $to   = $today;
if ($from > $to) { $tmp=$from; $from=$to; $to=$tmp; }

$generatedAt = date('Y-m-d H:i');

/* -----------------------------
  Lookups
------------------------------ */
$catsSt = $pdo->prepare("SELECT id,type,name FROM categories WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY type, sort_order, name");
$catsSt->execute([$uid]);
$categories = $catsSt->fetchAll();

$accSt = $pdo->prepare("SELECT id,name,currency,opening_balance FROM accounts WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$accSt->execute([$uid]);
$accounts = $accSt->fetchAll();

$cardsSt = $pdo->prepare("SELECT id,name,currency,card_limit,opening_debt FROM credit_cards WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$cardsSt->execute([$uid]);
$cards = $cardsSt->fetchAll();

$invSt = $pdo->prepare("SELECT id,name,currency FROM investment_accounts WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$invSt->execute([$uid]);
$investments = $invSt->fetchAll();

/* -----------------------------
  Range totals (income/expense/net)
------------------------------ */
$sumSt = $pdo->prepare("
  SELECT
    COALESCE(SUM(CASE WHEN kind='income'  THEN amount ELSE 0 END),0) AS income_sum,
    COALESCE(SUM(CASE WHEN kind='expense' THEN amount ELSE 0 END),0) AS expense_sum
  FROM transactions
  WHERE user_id=? AND deleted_at IS NULL
    AND occurred_on BETWEEN ? AND ?
");
$sumSt->execute([$uid, $from, $to]);
$sumRow = $sumSt->fetch() ?: ['income_sum'=>0,'expense_sum'=>0];

$incomeSum = (float)$sumRow['income_sum'];
$expenseSum = (float)$sumRow['expense_sum'];
$netSum = $incomeSum - $expenseSum;

/* -----------------------------
  Current balances (as of $to)
  Accounts: opening + (incomes to account) - (expenses from account) + transfers in/out
------------------------------ */
$accBal = [];
$totalBalance = 0.0;

$accBalSt = $pdo->prepare("
  SELECT
    a.id, a.name, a.opening_balance,
    COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.occurred_on<=?
      AND t.kind='income' AND t.dest_type='account' AND t.dest_id=a.id THEN t.amount ELSE 0 END),0) AS inc_in,
    COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.occurred_on<=?
      AND t.kind='expense' AND t.source_type='account' AND t.source_id=a.id THEN t.amount ELSE 0 END),0) AS exp_out,
    COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.occurred_on<=?
      AND t.kind='transfer' AND t.dest_type='account' AND t.dest_id=a.id THEN t.amount ELSE 0 END),0) AS tr_in,
    COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.occurred_on<=?
      AND t.kind='transfer' AND t.source_type='account' AND t.source_id=a.id THEN t.amount ELSE 0 END),0) AS tr_out
  FROM accounts a
  LEFT JOIN transactions t ON t.user_id=a.user_id
  WHERE a.user_id=? AND a.deleted_at IS NULL AND a.is_active=1
  GROUP BY a.id
  ORDER BY a.name
");
$accBalSt->execute([$to,$to,$to,$to,$uid]);
$accBalRows = $accBalSt->fetchAll();

foreach ($accBalRows as $r) {
  $bal = (float)$r['opening_balance'] + (float)$r['inc_in'] - (float)$r['exp_out'] + (float)$r['tr_in'] - (float)$r['tr_out'];
  $accBal[] = [
    'id' => (int)$r['id'],
    'name' => (string)$r['name'],
    'balance' => $bal,
  ];
  $totalBalance += $bal;
}

/* -----------------------------
  Card debt (as of $to)
  debt = opening + card expenses + card->out transfers - payments (to card)
------------------------------ */
$cardDebtRows = [];
$totalCardDebt = 0.0;

$cardDebtSt = $pdo->prepare("
  SELECT
    c.id, c.name, c.card_limit, c.opening_debt,
    COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.occurred_on<=?
      AND t.kind='expense' AND t.source_type='credit_card' AND t.source_id=c.id THEN t.amount ELSE 0 END),0) AS card_exp,
    COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.occurred_on<=?
      AND t.kind='transfer' AND t.dest_type='credit_card' AND t.dest_id=c.id THEN t.amount ELSE 0 END),0) AS pay_to_card,
    COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.occurred_on<=?
      AND t.kind='transfer' AND t.source_type='credit_card' AND t.source_id=c.id THEN t.amount ELSE 0 END),0) AS card_out
  FROM credit_cards c
  LEFT JOIN transactions t ON t.user_id=c.user_id
  WHERE c.user_id=? AND c.deleted_at IS NULL AND c.is_active=1
  GROUP BY c.id
  ORDER BY c.name
");
$cardDebtSt->execute([$to,$to,$to,$uid]);
$rows = $cardDebtSt->fetchAll();

foreach ($rows as $r) {
  $debt = (float)$r['opening_debt'] + (float)$r['card_exp'] + (float)$r['card_out'] - (float)$r['pay_to_card'];
  $limit = (float)$r['card_limit'];
  $remaining = $limit - $debt;
  $cardDebtRows[] = [
    'id' => (int)$r['id'],
    'name' => (string)$r['name'],
    'debt' => $debt,
    'limit' => $limit,
    'remaining' => $remaining,
  ];
  $totalCardDebt += $debt;
}

/* -----------------------------
  Investment portfolio (cash-based as of $to)
  in = transfers to investment, out = transfers from investment
------------------------------ */
$invNetRows = [];
$portfolioValue = 0.0;

$invNetSt = $pdo->prepare("
  SELECT
    ia.id, ia.name,
    COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.occurred_on<=?
      AND t.kind='transfer' AND t.dest_type='investment' AND t.dest_id=ia.id THEN t.amount ELSE 0 END),0) AS inv_in,
    COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.occurred_on<=?
      AND t.kind='transfer' AND t.source_type='investment' AND t.source_id=ia.id THEN t.amount ELSE 0 END),0) AS inv_out
  FROM investment_accounts ia
  LEFT JOIN transactions t ON t.user_id=ia.user_id
  WHERE ia.user_id=? AND ia.deleted_at IS NULL AND ia.is_active=1
  GROUP BY ia.id
  ORDER BY ia.name
");
$invNetSt->execute([$to,$to,$uid]);
$invRows = $invNetSt->fetchAll();

foreach ($invRows as $r) {
  $net = (float)$r['inv_in'] - (float)$r['inv_out'];
  $invNetRows[] = ['id'=>(int)$r['id'], 'name'=>(string)$r['name'], 'net'=>$net];
  $portfolioValue += $net;
}

/* -----------------------------
  Trend (monthly) within selected range
------------------------------ */
$trendSt = $pdo->prepare("
  SELECT DATE_FORMAT(occurred_on,'%Y-%m') AS ym,
         COALESCE(SUM(CASE WHEN kind='income' THEN amount ELSE 0 END),0) AS inc,
         COALESCE(SUM(CASE WHEN kind='expense' THEN amount ELSE 0 END),0) AS exp
  FROM transactions
  WHERE user_id=? AND deleted_at IS NULL
    AND occurred_on BETWEEN ? AND ?
  GROUP BY ym
  ORDER BY ym
");
$trendSt->execute([$uid, $from, $to]);
$trend = $trendSt->fetchAll();

$maxExpMonth = null;
$bestNetMonth = null;
$maxExpVal = -1.0;
$bestNetVal = -INF;

foreach ($trend as $t) {
  $exp = (float)$t['exp'];
  $net = (float)$t['inc'] - (float)$t['exp'];
  if ($exp > $maxExpVal) { $maxExpVal = $exp; $maxExpMonth = (string)$t['ym']; }
  if ($net > $bestNetVal) { $bestNetVal = $net; $bestNetMonth = (string)$t['ym']; }
}

/* -----------------------------
  Top categories (range)
------------------------------ */
$topCatSt = $pdo->prepare("
  SELECT c.name, COALESCE(SUM(t.amount),0) AS total
  FROM transactions t
  LEFT JOIN categories c ON c.id=t.category_id
  WHERE t.user_id=? AND t.deleted_at IS NULL
    AND t.occurred_on BETWEEN ? AND ?
    AND t.kind=?
  GROUP BY c.name
  ORDER BY total DESC
  LIMIT 1
");

$topExpenseCat = ['name'=>'—','total'=>0.0];
$topIncomeCat = ['name'=>'—','total'=>0.0];

$topCatSt->execute([$uid,$from,$to,'expense']);
$r = $topCatSt->fetch();
if ($r) $topExpenseCat = ['name'=>(string)($r['name'] ?? '—'), 'total'=>(float)$r['total']];

$topCatSt->execute([$uid,$from,$to,'income']);
$r = $topCatSt->fetch();
if ($r) $topIncomeCat = ['name'=>(string)($r['name'] ?? '—'), 'total'=>(float)$r['total']];

/* -----------------------------
  Portfolio P/L (simple realized from trades in range)
  avg cost model per symbol (range-based)
------------------------------ */
$pl = 0.0;
$trSt = $pdo->prepare("
  SELECT side, asset_symbol, quantity, price, fee
  FROM investment_trades
  WHERE user_id=? AND deleted_at IS NULL
    AND traded_at BETWEEN ? AND ?
  ORDER BY traded_at ASC, id ASC
");
$trSt->execute([$uid, $from.' 00:00:00', $to.' 23:59:59']);
$trades = $trSt->fetchAll();

$pos = []; // symbol => ['qty'=>, 'cost'=>]
foreach ($trades as $tr) {
  $side = (string)$tr['side']; // buy/sell
  $sym  = strtoupper((string)$tr['asset_symbol']);
  $qty  = (float)$tr['quantity'];
  $price= (float)$tr['price'];
  $fee  = (float)($tr['fee'] ?? 0);

  if ($qty <= 0) continue;
  if (!isset($pos[$sym])) $pos[$sym] = ['qty'=>0.0,'cost'=>0.0];

  if ($side === 'buy') {
    $pos[$sym]['qty']  += $qty;
    $pos[$sym]['cost'] += ($qty * $price) + $fee;
  } elseif ($side === 'sell') {
    // average cost
    $heldQty = $pos[$sym]['qty'];
    if ($heldQty <= 0) continue;
    $avgCost = $pos[$sym]['cost'] / max($heldQty, 1e-9);

    $sellQty = min($qty, $heldQty);
    $proceeds = ($sellQty * $price) - $fee;
    $costPart = $sellQty * $avgCost;

    $pl += ($proceeds - $costPart);

    $pos[$sym]['qty']  -= $sellQty;
    $pos[$sym]['cost'] -= $costPart;
  }
}

/* -----------------------------
  Recent transactions (last 10 overall)
------------------------------ */
$recentSt = $pdo->prepare("
  SELECT id, occurred_on, kind, amount, description, category_id, source_type, source_id, dest_type, dest_id
  FROM transactions
  WHERE user_id=? AND deleted_at IS NULL
  ORDER BY occurred_on DESC, id DESC
  LIMIT 10
");
$recentSt->execute([$uid]);
$recent = $recentSt->fetchAll();

function cat_name(array $cats, ?int $id): string {
  if (!$id) return '—';
  foreach ($cats as $c) if ((int)$c['id']===$id) return (string)$c['name'];
  return '—';
}
function name_by_id(array $rows, int $id): string {
  foreach ($rows as $r) if ((int)$r['id']===$id) return (string)$r['name'];
  return '—';
}
function tx_detail(array $t, array $cats, array $accs, array $cards, array $invs): string {
  $kind = (string)$t['kind'];
  $cat = cat_name($cats, $t['category_id'] ? (int)$t['category_id'] : null);

  $srcT=(string)$t['source_type']; $dstT=(string)$t['dest_type'];
  $srcId = $t['source_id'] ? (int)$t['source_id'] : 0;
  $dstId = $t['dest_id'] ? (int)$t['dest_id'] : 0;

  $src='—'; $dst='—';
  if ($srcT==='account') $src=name_by_id($accs,$srcId);
  if ($srcT==='credit_card') $src=name_by_id($cards,$srcId);
  if ($srcT==='investment') $src=name_by_id($invs,$srcId);

  if ($dstT==='account') $dst=name_by_id($accs,$dstId);
  if ($dstT==='credit_card') $dst=name_by_id($cards,$dstId);
  if ($dstT==='investment') $dst=name_by_id($invs,$dstId);

  $desc = trim((string)($t['description'] ?? ''));

  if ($kind==='transfer') {
    $base = "Transfer • {$src} → {$dst}";
    return $desc ? ($base." — ".$desc) : $base;
  }

  $where = $srcT!=='none' ? $src : ($dstT!=='none' ? $dst : '—');
  $base = "{$cat} • {$where}";
  return $desc ? ($base." — ".$desc) : $base;
}

/* -----------------------------
  Goals: last 3 (active)
------------------------------ */
$goalsSt = $pdo->prepare("
  SELECT g.id, g.name, g.target_amount, g.target_date, g.status,
         COALESCE(SUM(CASE WHEN gc.deleted_at IS NULL THEN gc.amount ELSE 0 END),0) AS saved_amount
  FROM goals g
  LEFT JOIN goal_contributions gc
    ON gc.goal_id=g.id AND gc.user_id=g.user_id
  WHERE g.user_id=? AND g.deleted_at IS NULL
    AND g.status IN ('active','paused')
  GROUP BY g.id
  ORDER BY g.target_date ASC
  LIMIT 3
");
$goalsSt->execute([$uid]);
$goals = $goalsSt->fetchAll();

/* -----------------------------
  Smart Insights
------------------------------ */
$insights = [];

// 1) Top expense category insight
if ($topExpenseCat['name'] !== '—' && $topExpenseCat['total'] > 0) {
  $insights[] = "Bu dönemde en çok harcama yaptığın kategori: <b>".h($topExpenseCat['name'])."</b> (".money_try($topExpenseCat['total']).").";
}

// 2) Goal deadlines within 10 days
foreach ($goals as $g) {
  $daysLeft = (int)floor((strtotime((string)$g['target_date']) - strtotime(date('Y-m-d'))) / 86400);
  if ($daysLeft >= 0 && $daysLeft <= 10) {
    $insights[] = "Hedef <b>".h((string)$g['name'])."</b> için son <b>{$daysLeft}</b> gün kaldı.";
  }
}

// 3) Expense higher than income in range
if ($expenseSum > $incomeSum && ($incomeSum + $expenseSum) > 0) {
  $insights[] = "Bu dönemde giderlerin gelirlerinden yüksek görünüyor (Net: <b>".money_try($netSum)."</b>).";
}

// 4) Card debt warning
if ($totalCardDebt > 0) {
  $insights[] = "Toplam kart borcun: <b>".money_try($totalCardDebt)."</b>. Ödeme planı yapman iyi olabilir.";
}

require __DIR__ . '/../inc/header.php';
?>

<!-- Dashboard Header -->
<div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
  <div class="flex flex-col md:flex-row md:items-center justify-between gap-3">
    <div>
      <h2 class="text-lg font-semibold">📌 Dashboard</h2>
      <p class="text-sm opacity-70">Özet, trendler, öne çıkanlar ve hızlı içgörüler.</p>
    </div>

    <form method="get" class="flex flex-wrap gap-2 items-end">
      <div>
        <label class="text-xs opacity-70">Başlangıç</label>
        <input type="date" name="from" value="<?= h($from) ?>"
               class="mt-1 px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
      </div>
      <div>
        <label class="text-xs opacity-70">Bitiş</label>
        <input type="date" name="to" value="<?= h($to) ?>"
               class="mt-1 px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
      </div>
      <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">Uygula</button>
    </form>
  </div>
</div>

<!-- Top Summary Cards -->
<div class="mt-4 grid grid-cols-1 md:grid-cols-2 xl:grid-cols-6 gap-4">
  <div class="p-4 rounded-2xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <p class="text-sm opacity-70">Gelir</p>
    <p class="text-xl font-bold"><?= money_try($incomeSum) ?></p>
    <p class="text-xs opacity-60 mt-1"><?= h($from) ?> → <?= h($to) ?></p>
  </div>

  <div class="p-4 rounded-2xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <p class="text-sm opacity-70">Gider</p>
    <p class="text-xl font-bold"><?= money_try($expenseSum) ?></p>
    <p class="text-xs opacity-60 mt-1"><?= h($from) ?> → <?= h($to) ?></p>
  </div>

  <div class="p-4 rounded-2xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <p class="text-sm opacity-70">Net</p>
    <p class="text-xl font-bold"><?= money_try($netSum) ?></p>
    <p class="text-xs opacity-60 mt-1">Gelir − Gider</p>
  </div>

  <div class="p-4 rounded-2xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <p class="text-sm opacity-70">Toplam Bakiye</p>
    <p class="text-xl font-bold"><?= money_try($totalBalance) ?></p>
    <p class="text-xs opacity-60 mt-1">Hesaplar (<?= h($to) ?>)</p>
  </div>

  <div class="p-4 rounded-2xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <p class="text-sm opacity-70">Kart Borcu</p>
    <p class="text-xl font-bold"><?= money_try($totalCardDebt) ?></p>
    <p class="text-xs opacity-60 mt-1">Kartlar (<?= h($to) ?>)</p>
  </div>

  <div class="p-4 rounded-2xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <p class="text-sm opacity-70">Portföy Değeri</p>
    <p class="text-xl font-bold"><?= money_try($portfolioValue) ?></p>
    <p class="text-xs opacity-60 mt-1">Yatırım (nakit bazlı)</p>
  </div>
</div>

<!-- Trend + Highlights -->
<div class="mt-4 grid grid-cols-1 xl:grid-cols-3 gap-4">
  <!-- Trend -->
  <div class="xl:col-span-2 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <div class="flex items-center justify-between">
      <h3 class="text-lg font-semibold">📈 Aylık Trend</h3>
      <div class="text-xs opacity-70">Seçilen aralık: <?= h($from) ?> → <?= h($to) ?></div>
    </div>

    <div class="mt-3 overflow-x-auto">
      <table class="w-full text-sm">
        <thead class="text-xs uppercase opacity-70">
          <tr class="border-b border-slate-200 dark:border-slate-800">
            <th class="text-left py-2 pr-3">Ay</th>
            <th class="text-right py-2 pl-3">Gelir</th>
            <th class="text-right py-2 pl-3">Gider</th>
            <th class="text-right py-2 pl-3">Net</th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$trend): ?>
            <tr><td colspan="4" class="py-10 text-center opacity-70">Bu aralıkta trend verisi yok.</td></tr>
          <?php endif; ?>

          <?php foreach ($trend as $t): ?>
            <?php
              $inc=(float)$t['inc']; $exp=(float)$t['exp']; $net=$inc-$exp;
            ?>
            <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
              <td class="py-3 pr-3 font-semibold"><?= h((string)$t['ym']) ?></td>
              <td class="py-3 pl-3 text-right"><?= money_try($inc) ?></td>
              <td class="py-3 pl-3 text-right"><?= money_try($exp) ?></td>
              <td class="py-3 pl-3 text-right font-semibold"><?= money_try($net) ?></td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>

  <!-- Highlights -->
  <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <h3 class="text-lg font-semibold">⭐ Öne Çıkanlar</h3>
    <p class="text-sm opacity-70 mt-1">Seçilen döneme göre</p>

    <div class="mt-4 space-y-3 text-sm">
      <div class="flex items-center justify-between gap-3">
        <span class="opacity-80">En fazla harcama ayı</span>
        <span class="font-semibold"><?= $maxExpMonth ? h($maxExpMonth) : '—' ?></span>
      </div>

      <div class="flex items-center justify-between gap-3">
        <span class="opacity-80">En kârlı ay</span>
        <span class="font-semibold"><?= $bestNetMonth ? h($bestNetMonth) : '—' ?></span>
      </div>

      <div class="flex items-center justify-between gap-3">
        <span class="opacity-80">En çok harcama kategorisi</span>
        <span class="font-semibold"><?= h((string)$topExpenseCat['name']) ?></span>
      </div>

      <div class="flex items-center justify-between gap-3">
        <span class="opacity-80">En çok gelir kategorisi</span>
        <span class="font-semibold"><?= h((string)$topIncomeCat['name']) ?></span>
      </div>

      <div class="flex items-center justify-between gap-3">
        <span class="opacity-80">Portföy Kâr/Zarar</span>
        <span class="font-semibold"><?= money_try($pl) ?></span>
      </div>

      <div class="text-xs opacity-70 pt-2">
        Not: Portföy değerinde piyasa fiyatı yok; “nakit aktarım” bazlı.
        K/Z ise trade tablosundan (basit ortalama maliyet) hesaplanır.
      </div>
    </div>
  </div>
</div>

<!-- Distributions -->
<div class="mt-4 grid grid-cols-1 xl:grid-cols-3 gap-4">
  <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <h3 class="text-lg font-semibold">🏦 Bakiye Dağılımı</h3>
    <div class="mt-3 overflow-x-auto">
      <table class="w-full text-sm">
        <thead class="text-xs uppercase opacity-70">
          <tr class="border-b border-slate-200 dark:border-slate-800">
            <th class="text-left py-2 pr-3">Hesap</th>
            <th class="text-right py-2 pl-3">Bakiye</th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$accBal): ?>
            <tr><td colspan="2" class="py-10 text-center opacity-70">Hesap yok.</td></tr>
          <?php endif; ?>
          <?php foreach ($accBal as $a): ?>
            <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
              <td class="py-3 pr-3 font-semibold"><?= h($a['name']) ?></td>
              <td class="py-3 pl-3 text-right"><?= money_try((float)$a['balance']) ?></td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>

  <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <h3 class="text-lg font-semibold">💳 Kart Borç Dağılımı</h3>
    <div class="mt-3 overflow-x-auto">
      <table class="w-full text-sm">
        <thead class="text-xs uppercase opacity-70">
          <tr class="border-b border-slate-200 dark:border-slate-800">
            <th class="text-left py-2 pr-3">Kart</th>
            <th class="text-right py-2 pl-3">Borç</th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$cardDebtRows): ?>
            <tr><td colspan="2" class="py-10 text-center opacity-70">Kart yok.</td></tr>
          <?php endif; ?>
          <?php foreach ($cardDebtRows as $c): ?>
            <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
              <td class="py-3 pr-3 font-semibold"><?= h($c['name']) ?></td>
              <td class="py-3 pl-3 text-right"><?= money_try((float)$c['debt']) ?></td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>

  <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <h3 class="text-lg font-semibold">📈 Yatırım Dağılımı</h3>
    <div class="mt-3 overflow-x-auto">
      <table class="w-full text-sm">
        <thead class="text-xs uppercase opacity-70">
          <tr class="border-b border-slate-200 dark:border-slate-800">
            <th class="text-left py-2 pr-3">Hesap</th>
            <th class="text-right py-2 pl-3">Net Aktarım</th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$invNetRows): ?>
            <tr><td colspan="2" class="py-10 text-center opacity-70">Yatırım hesabı yok.</td></tr>
          <?php endif; ?>
          <?php foreach ($invNetRows as $i): ?>
            <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
              <td class="py-3 pr-3 font-semibold"><?= h($i['name']) ?></td>
              <td class="py-3 pl-3 text-right"><?= money_try((float)$i['net']) ?></td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<!-- Recent transactions + Goals + Insights -->
<div class="mt-4 grid grid-cols-1 xl:grid-cols-3 gap-4">
  <div class="xl:col-span-2 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <div class="flex items-center justify-between gap-3">
      <h3 class="text-lg font-semibold">🧾 Son İşlemler</h3>
      <a class="text-sm underline opacity-80" href="<?= h(page_url('reports.php')) ?>">Raporlara Git</a>
    </div>

    <div class="mt-3 overflow-x-auto">
      <table class="w-full text-sm">
        <thead class="text-xs uppercase opacity-70">
          <tr class="border-b border-slate-200 dark:border-slate-800">
            <th class="text-left py-2 pr-3">Tarih</th>
            <th class="text-left py-2 pr-3">Tür</th>
            <th class="text-left py-2 pr-3">Detay</th>
            <th class="text-right py-2 pl-3">Tutar</th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$recent): ?>
            <tr><td colspan="4" class="py-10 text-center opacity-70">Henüz işlem yok.</td></tr>
          <?php endif; ?>

          <?php foreach ($recent as $t): ?>
            <?php
              $k = (string)$t['kind'];
              $label = $k==='income' ? 'Gelir' : ($k==='expense' ? 'Gider' : 'Transfer');
              $detail = tx_detail($t, $categories, $accBalRows, $cardDebtRows, $invNetRows);
              $amt = (float)$t['amount'];
              $signed = $k==='income' ? ('+'.money_try($amt)) : ($k==='expense' ? ('-'.money_try($amt)) : money_try($amt));
            ?>
            <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
              <td class="py-3 pr-3"><?= h((string)$t['occurred_on']) ?></td>
              <td class="py-3 pr-3"><?= h($label) ?></td>
              <td class="py-3 pr-3"><?= h($detail) ?></td>
              <td class="py-3 pl-3 text-right font-semibold"><?= h($signed) ?></td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>

  <div class="space-y-4">
    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
      <h3 class="text-lg font-semibold">🎯 Son 3 Hedef</h3>

      <div class="mt-3 space-y-3">
        <?php if (!$goals): ?>
          <div class="text-sm opacity-70">Hedef yok.</div>
        <?php endif; ?>

        <?php foreach ($goals as $g): ?>
          <?php
            $target = (float)$g['target_amount'];
            $saved  = (float)$g['saved_amount'];
            $pct = $target > 0 ? max(0, min(100, ($saved/$target)*100)) : 0;
            $daysLeft = (int)floor((strtotime((string)$g['target_date']) - strtotime(date('Y-m-d'))) / 86400);
            $daysText = $daysLeft >= 0 ? ($daysLeft.' gün kaldı') : (abs($daysLeft).' gün geçti');
          ?>
          <div class="p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <div class="flex items-start justify-between gap-3">
              <div>
                <div class="font-semibold"><?= h((string)$g['name']) ?></div>
                <div class="text-xs opacity-70 mt-1">
                  <?= money_try($saved) ?> / <?= money_try($target) ?> • <?= h((string)$g['target_date']) ?> • <?= h($daysText) ?>
                </div>
              </div>
              <div class="text-sm font-semibold">%<?= number_format($pct, 0, ',', '.') ?></div>
            </div>
            <div class="mt-3 h-2 rounded-full bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 overflow-hidden">
              <div class="h-full bg-primary" style="width: <?= (float)$pct ?>%"></div>
            </div>
          </div>
        <?php endforeach; ?>
      </div>

      <a class="inline-block mt-3 text-sm underline opacity-80" href="<?= h(page_url('goals.php')) ?>">Hedeflere Git</a>
    </div>

    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
      <h3 class="text-lg font-semibold">🧠 Akıllı İçgörüler</h3>
      <p class="text-sm opacity-70 mt-1">Kısa uyarılar ve özetler</p>

      <div class="mt-3 space-y-2 text-sm">
        <?php if (!$insights): ?>
          <div class="opacity-70">Şimdilik içgörü yok. Biraz veri girince burada öneriler çıkacak.</div>
        <?php endif; ?>

        <?php foreach ($insights as $msg): ?>
          <div class="p-3 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <?= $msg ?>
          </div>
        <?php endforeach; ?>
      </div>

      <div class="text-xs opacity-70 mt-3">
        Oluşturma zamanı: <?= h($generatedAt) ?>
      </div>
    </div>
  </div>
</div>

<?php require __DIR__ . '/../inc/footer.php'; ?>
