<?php
declare(strict_types=1);

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/functions.php';

$uid = current_user_id() ?? 1;

// --- Load dropdown data (active only, not deleted) ---
$catsIncome = $pdo->prepare("SELECT id,name FROM categories WHERE user_id=? AND type='income' AND deleted_at IS NULL AND is_active=1 ORDER BY sort_order, name");
$catsIncome->execute([$uid]);
$catsIncome = $catsIncome->fetchAll();

$catsExpense = $pdo->prepare("SELECT id,name FROM categories WHERE user_id=? AND type='expense' AND deleted_at IS NULL AND is_active=1 ORDER BY sort_order, name");
$catsExpense->execute([$uid]);
$catsExpense = $catsExpense->fetchAll();

$accounts = $pdo->prepare("SELECT id,name,currency FROM accounts WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$accounts->execute([$uid]);
$accounts = $accounts->fetchAll();

$cards = $pdo->prepare("SELECT id,name,currency FROM credit_cards WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$cards->execute([$uid]);
$cards = $cards->fetchAll();

// --- Actions ---
$err = '';
$ok  = '';

$action = (string)($_POST['action'] ?? '');

if ($action === 'create') {
  $kind = (string)($_POST['kind'] ?? 'expense'); // income|expense|transfer
  $date = (string)($_POST['occurred_on'] ?? date('Y-m-d'));
  $amount = (float)($_POST['amount'] ?? 0);
  $desc = trim((string)($_POST['description'] ?? ''));
  $currency = (string)($_POST['currency'] ?? 'TRY');

  if ($amount <= 0) {
    $err = "Tutar 0'dan büyük olmalı.";
  } else {
    $categoryId = null;
    $sourceType = 'none';
    $sourceId   = null;
    $destType   = 'none';
    $destId     = null;

    if ($kind === 'income') {
      $categoryId = (int)($_POST['category_income'] ?? 0);
      $sourceType = 'none';
      $destType = 'account';
      $destId = (int)($_POST['dest_account'] ?? 0);

      if ($categoryId <= 0) $err = "Gelir kategorisi seç.";
      if (!$err && $destId <= 0) $err = "Gelir için hedef hesabı seç.";
    }

    if ($kind === 'expense') {
      $categoryId = (int)($_POST['category_expense'] ?? 0);
      $paySource = (string)($_POST['expense_source'] ?? 'account'); // account|credit_card

      if ($categoryId <= 0) $err = "Gider kategorisi seç.";

      if (!$err && $paySource === 'account') {
        $sourceType = 'account';
        $sourceId = (int)($_POST['expense_account'] ?? 0);
        if ($sourceId <= 0) $err = "Gider için hesap seç.";
      }

      if (!$err && $paySource === 'credit_card') {
        $sourceType = 'credit_card';
        $sourceId = (int)($_POST['expense_card'] ?? 0);
        if ($sourceId <= 0) $err = "Gider için kredi kartı seç.";
      }
    }

    if ($kind === 'transfer') {
      // MVP: account -> account
      $sourceType = 'account';
      $destType = 'account';
      $sourceId = (int)($_POST['tr_source_account'] ?? 0);
      $destId   = (int)($_POST['tr_dest_account'] ?? 0);

      if ($sourceId <= 0) $err = "Gönderici hesabı seç.";
      if (!$err && $destId <= 0) $err = "Hedef hesabı seç.";
      if (!$err && $sourceId === $destId) $err = "Gönderici ve hedef aynı olamaz.";
    }

    if (!$err) {
      $ins = $pdo->prepare("
        INSERT INTO transactions
          (user_id, kind, occurred_on, amount, currency, category_id, description, source_type, source_id, dest_type, dest_id)
        VALUES
          (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      ");
      $ins->execute([
        $uid, $kind, $date, $amount, $currency,
        $categoryId ?: null,
        $desc ?: null,
        $sourceType, $sourceId,
        $destType, $destId
      ]);
      $ok = "Kayıt eklendi ✅";
    }
  }
}

if ($action === 'trash') {
  $id = (int)($_POST['id'] ?? 0);
  if ($id > 0) {
    $st = $pdo->prepare("UPDATE transactions SET deleted_at=NOW() WHERE id=? AND user_id=? AND deleted_at IS NULL");
    $st->execute([$id, $uid]);
    $ok = "Kayıt çöp kutusuna taşındı 🗑️";
  }
}

// --- List latest transactions ---
$list = $pdo->prepare("
  SELECT t.*,
         c.name AS category_name
  FROM transactions t
  LEFT JOIN categories c ON c.id=t.category_id
  WHERE t.user_id=? AND t.deleted_at IS NULL
  ORDER BY t.occurred_on DESC, t.id DESC
  LIMIT 50
");
$list->execute([$uid]);
$rows = $list->fetchAll();

require __DIR__ . '/../inc/header.php';
?>

<div class="grid grid-cols-1 xl:grid-cols-3 gap-4">

  <!-- FORM -->
  <div class="xl:col-span-1 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <h2 class="text-lg font-semibold">🧾 İşlem Ekle</h2>
    <p class="text-sm opacity-70 mt-1">Gelir / Gider / Transfer</p>

    <?php if ($err): ?>
      <div class="mt-3 p-3 rounded-xl bg-rose-50 dark:bg-rose-950/40 border border-rose-200 dark:border-rose-900 text-sm">
        <?= h($err) ?>
      </div>
    <?php endif; ?>

    <?php if ($ok): ?>
      <div class="mt-3 p-3 rounded-xl bg-emerald-50 dark:bg-emerald-950/40 border border-emerald-200 dark:border-emerald-900 text-sm">
        <?= h($ok) ?>
      </div>
    <?php endif; ?>

    <form class="mt-4 space-y-3" method="post">
      <input type="hidden" name="action" value="create">

      <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Tür</label>
          <select id="kind" name="kind"
            class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="expense">Gider</option>
            <option value="income">Gelir</option>
            <option value="transfer">Transfer</option>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tarih</label>
          <input name="occurred_on" type="date" value="<?= h(date('Y-m-d')) ?>"
            class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div>
          <label class="text-sm opacity-70">Tutar</label>
          <input name="amount" type="number" step="0.01" placeholder="0.00"
            class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Para Birimi</label>
          <select name="currency"
            class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="TRY">TRY (₺)</option>
            <option value="USD">USD ($)</option>
            <option value="EUR">EUR (€)</option>
          </select>
        </div>
      </div>

      <!-- INCOME -->
      <div id="incomeBox" class="hidden p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label class="text-sm opacity-70">Kategori</label>
            <select name="category_income"
              class="mt-1 w-full px-3 py-2 rounded-xl bg-white/80 dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
              <option value="0">Seç...</option>
              <?php foreach($catsIncome as $c): ?>
                <option value="<?= (int)$c['id'] ?>"><?= h($c['name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div>
            <label class="text-sm opacity-70">Hedef Hesap</label>
            <select name="dest_account"
              class="mt-1 w-full px-3 py-2 rounded-xl bg-white/80 dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
              <option value="0">Seç...</option>
              <?php foreach($accounts as $a): ?>
                <option value="<?= (int)$a['id'] ?>"><?= h($a['name']) ?> (<?= h($a['currency']) ?>)</option>
              <?php endforeach; ?>
            </select>
          </div>
        </div>
      </div>

      <!-- EXPENSE -->
      <div id="expenseBox" class="p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label class="text-sm opacity-70">Kategori</label>
            <select name="category_expense"
              class="mt-1 w-full px-3 py-2 rounded-xl bg-white/80 dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
              <option value="0">Seç...</option>
              <?php foreach($catsExpense as $c): ?>
                <option value="<?= (int)$c['id'] ?>"><?= h($c['name']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div>
            <label class="text-sm opacity-70">Kaynak Türü</label>
            <select id="expenseSource" name="expense_source"
              class="mt-1 w-full px-3 py-2 rounded-xl bg-white/80 dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
              <option value="account">Hesap</option>
              <option value="credit_card">Kredi Kartı</option>
            </select>
          </div>

          <div id="expenseAccountBox">
            <label class="text-sm opacity-70">Hesap</label>
            <select name="expense_account"
              class="mt-1 w-full px-3 py-2 rounded-xl bg-white/80 dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
              <option value="0">Seç...</option>
              <?php foreach($accounts as $a): ?>
                <option value="<?= (int)$a['id'] ?>"><?= h($a['name']) ?> (<?= h($a['currency']) ?>)</option>
              <?php endforeach; ?>
            </select>
          </div>

          <div id="expenseCardBox" class="hidden">
            <label class="text-sm opacity-70">Kredi Kartı</label>
            <select name="expense_card"
              class="mt-1 w-full px-3 py-2 rounded-xl bg-white/80 dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
              <option value="0">Seç...</option>
              <?php foreach($cards as $k): ?>
                <option value="<?= (int)$k['id'] ?>"><?= h($k['name']) ?> (<?= h($k['currency']) ?>)</option>
              <?php endforeach; ?>
            </select>
          </div>
        </div>
      </div>

      <!-- TRANSFER -->
      <div id="transferBox" class="hidden p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label class="text-sm opacity-70">Gönderici Hesap</label>
            <select name="tr_source_account"
              class="mt-1 w-full px-3 py-2 rounded-xl bg-white/80 dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
              <option value="0">Seç...</option>
              <?php foreach($accounts as $a): ?>
                <option value="<?= (int)$a['id'] ?>"><?= h($a['name']) ?> (<?= h($a['currency']) ?>)</option>
              <?php endforeach; ?>
            </select>
          </div>

          <div>
            <label class="text-sm opacity-70">Hedef Hesap</label>
            <select name="tr_dest_account"
              class="mt-1 w-full px-3 py-2 rounded-xl bg-white/80 dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
              <option value="0">Seç...</option>
              <?php foreach($accounts as $a): ?>
                <option value="<?= (int)$a['id'] ?>"><?= h($a['name']) ?> (<?= h($a['currency']) ?>)</option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="md:col-span-2 text-xs opacity-70">
            Not: Transferler ileride kredi kartı ve yatırım türlerini de destekleyecek.
          </div>
        </div>
      </div>

      <div>
        <label class="text-sm opacity-70">Açıklama (opsiyonel)</label>
        <input name="description" placeholder="Örn: Freelance ödeme, kira..."
          class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
      </div>

      <button class="w-full px-4 py-2 rounded-xl bg-primary text-white font-semibold">
        ✅ Kaydet
      </button>
    </form>
  </div>

  <!-- LIST -->
  <div class="xl:col-span-2 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <div class="flex items-center justify-between gap-3">
      <div>
        <h2 class="text-lg font-semibold">Son İşlemler</h2>
        <p class="text-sm opacity-70">Son 50 kayıt (çöpe taşınmayanlar)</p>
      </div>
      <a href="<?= h(page_url('trash.php')) ?>"
         class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
        🗑️ Çöp Kutusu
      </a>
    </div>

    <div class="mt-4 overflow-x-auto">
      <table class="w-full text-sm">
        <thead class="text-xs uppercase opacity-70">
        <tr class="border-b border-slate-200 dark:border-slate-800">
          <th class="text-left py-2 pr-3">Tarih</th>
          <th class="text-left py-2 pr-3">Tür</th>
          <th class="text-left py-2 pr-3">Kategori</th>
          <th class="text-left py-2 pr-3">Açıklama</th>
          <th class="text-right py-2 pl-3">Tutar</th>
          <th class="text-right py-2 pl-3">İşlem</th>
        </tr>
        </thead>
        <tbody>
        <?php if (!$rows): ?>
          <tr>
            <td colspan="6" class="py-6 text-center opacity-70">Henüz kayıt yok.</td>
          </tr>
        <?php endif; ?>

        <?php foreach($rows as $r): ?>
          <?php
            $sign = $r['kind'] === 'income' ? '+' : ($r['kind'] === 'expense' ? '-' : '');
            $kindLabel = $r['kind'] === 'income' ? 'Gelir' : ($r['kind'] === 'expense' ? 'Gider' : 'Transfer');
          ?>
          <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
            <td class="py-2 pr-3 whitespace-nowrap"><?= h($r['occurred_on']) ?></td>
            <td class="py-2 pr-3"><?= h($kindLabel) ?></td>
            <td class="py-2 pr-3"><?= h($r['category_name'] ?? '-') ?></td>
            <td class="py-2 pr-3"><?= h($r['description'] ?? '') ?></td>
            <td class="py-2 pl-3 text-right font-semibold whitespace-nowrap">
              <?= h($sign) ?><?= money((float)$r['amount'], $r['currency']==='TRY'?'₺':($r['currency']==='USD'?'$':'€')) ?>
            </td>
            <td class="py-2 pl-3 text-right">
              <form method="post" onsubmit="return confirm('Çöp kutusuna taşınsın mı?');">
                <input type="hidden" name="action" value="trash">
                <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                <button class="px-3 py-1 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
                  🗑️
                </button>
              </form>
            </td>
          </tr>
        <?php endforeach; ?>
        </tbody>
      </table>
    </div>

  </div>
</div>

<script>
  const kind = document.getElementById('kind');
  const incomeBox = document.getElementById('incomeBox');
  const expenseBox = document.getElementById('expenseBox');
  const transferBox = document.getElementById('transferBox');

  const expenseSource = document.getElementById('expenseSource');
  const expenseAccountBox = document.getElementById('expenseAccountBox');
  const expenseCardBox = document.getElementById('expenseCardBox');

  function renderKind(){
    const v = kind.value;
    incomeBox.classList.toggle('hidden', v !== 'income');
    expenseBox.classList.toggle('hidden', v !== 'expense');
    transferBox.classList.toggle('hidden', v !== 'transfer');
  }

  function renderExpenseSource(){
    const v = expenseSource.value;
    expenseAccountBox.classList.toggle('hidden', v !== 'account');
    expenseCardBox.classList.toggle('hidden', v !== 'credit_card');
  }

  kind.addEventListener('change', renderKind);
  expenseSource.addEventListener('change', renderExpenseSource);
  renderKind();
  renderExpenseSource();
</script>

<?php require __DIR__ . '/../inc/footer.php'; ?>
