<?php
// project_detail.php
require_once 'config.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

$project_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
$user_id = $_SESSION['user_id'];

// Proje bilgilerini getir
$sql = "SELECT p.*, u.full_name as creator_name 
        FROM projects p 
        LEFT JOIN users u ON p.created_by = u.id 
        WHERE p.id = ?";
$stmt = $conn->prepare($sql);
$stmt->bind_param("i", $project_id);
$stmt->execute();
$project = $stmt->get_result()->fetch_assoc();

if (!$project) {
    die("Proje bulunamadı!");
}

// Kullanıcının bu projede yetkisi var mı kontrol et
$check_member_sql = "SELECT * FROM project_members WHERE project_id = ? AND user_id = ?";
$check_stmt = $conn->prepare($check_member_sql);
$check_stmt->bind_param("ii", $project_id, $user_id);
$check_stmt->execute();
$is_member = $check_stmt->get_result()->num_rows > 0 || $project['created_by'] == $user_id;

if (!$is_member) {
    die("Bu projeyi görüntüleme yetkiniz yok!");
}

// Proje görevlerini getir
$tasks_sql = "SELECT t.*, u.full_name as assignee_name 
              FROM tasks t 
              LEFT JOIN users u ON t.assigned_to = u.id 
              WHERE t.project_id = ? 
              ORDER BY 
                CASE t.priority 
                    WHEN 'Kritik' THEN 1
                    WHEN 'Yüksek' THEN 2
                    WHEN 'Orta' THEN 3
                    WHEN 'Düşük' THEN 4
                END,
                t.end_date ASC";
$tasks_stmt = $conn->prepare($tasks_sql);
$tasks_stmt->bind_param("i", $project_id);
$tasks_stmt->execute();
$tasks = $tasks_stmt->get_result();

// Ekip üyelerini getir
$members_sql = "SELECT u.id, u.full_name, u.email, pm.role 
                FROM project_members pm 
                JOIN users u ON pm.user_id = u.id 
                WHERE pm.project_id = ? 
                ORDER BY 
                  CASE pm.role 
                      WHEN 'Yönetici' THEN 1
                      WHEN 'Üye' THEN 2
                      WHEN 'Gözlemci' THEN 3
                  END";
$members_stmt = $conn->prepare($members_sql);
$members_stmt->bind_param("i", $project_id);
$members_stmt->execute();
$members = $members_stmt->get_result();

// Proje istatistikleri
$stats_sql = "SELECT 
    COUNT(*) as total_tasks,
    SUM(CASE WHEN status = 'Tamamlandı' THEN 1 ELSE 0 END) as completed_tasks,
    SUM(CASE WHEN status != 'Tamamlandı' AND end_date < CURDATE() THEN 1 ELSE 0 END) as overdue_tasks,
    AVG(estimated_hours) as avg_estimation
    FROM tasks 
    WHERE project_id = ?";
$stats_stmt = $conn->prepare($stats_sql);
$stats_stmt->bind_param("i", $project_id);
$stats_stmt->execute();
$stats = $stats_stmt->get_result()->fetch_assoc();

$progress = $stats['total_tasks'] > 0 ? round(($stats['completed_tasks'] / $stats['total_tasks']) * 100) : 0;
?>

<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($project['project_name']); ?> - Proje Detay</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: Arial, sans-serif; background: #f5f5f5; }
        
        .header { background: white; padding: 15px 30px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); margin-bottom: 30px; }
        .header-content { display: flex; justify-content: space-between; align-items: center; }
        .logo { font-size: 24px; font-weight: bold; color: #2196F3; }
        .nav-links a { margin-left: 15px; color: #333; text-decoration: none; }
        
        .container { max-width: 1400px; margin: 0 auto; padding: 0 20px; }
        
        .project-header { background: white; padding: 25px; border-radius: 10px; margin-bottom: 30px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .project-title { font-size: 28px; color: #333; margin-bottom: 10px; }
        .project-meta { display: flex; gap: 30px; margin-bottom: 20px; color: #666; }
        .meta-item { display: flex; align-items: center; gap: 8px; }
        .project-description { background: #f9f9f9; padding: 20px; border-radius: 5px; margin-top: 20px; line-height: 1.6; }
        
        .status-badge, .priority-badge {
            display: inline-block; padding: 5px 15px; border-radius: 15px; font-size: 14px; font-weight: bold;
        }
        .status-planned { background: #ffeb3b; color: #333; }
        .status-devam-ediyor { background: #2196F3; color: white; }
        .status-tamamlandı { background: #4CAF50; color: white; }
        .status-beklemede { background: #ff9800; color: white; }
        .status-iptal-edildi { background: #f44336; color: white; }
        
        .priority-kritik { background: #d32f2f; color: white; }
        .priority-yüksek { background: #f44336; color: white; }
        .priority-orta { background: #ff9800; color: white; }
        .priority-düşük { background: #4CAF50; color: white; }
        
        .stats-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 30px; }
        .stat-card { background: white; padding: 20px; border-radius: 10px; box-shadow: 0 2px 5px rgba(0,0,0,0.1); text-align: center; }
        .stat-value { font-size: 32px; font-weight: bold; color: #2196F3; }
        .stat-label { color: #666; margin-top: 5px; }
        
        .progress-bar { height: 10px; background: #eee; border-radius: 5px; margin: 10px 0; overflow: hidden; }
        .progress-fill { height: 100%; background: #4CAF50; border-radius: 5px; transition: width 0.3s; }
        
        .content-grid { display: grid; grid-template-columns: 2fr 1fr; gap: 30px; }
        .section { background: white; padding: 25px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .section h3 { margin-bottom: 20px; color: #333; border-bottom: 2px solid #2196F3; padding-bottom: 10px; }
        
        .tasks-table { width: 100%; border-collapse: collapse; }
        .tasks-table th { text-align: left; padding: 12px; background: #f5f5f5; border-bottom: 2px solid #ddd; }
        .tasks-table td { padding: 12px; border-bottom: 1px solid #eee; }
        .tasks-table tr:hover { background: #f9f9f9; }
        
        .members-list { list-style: none; }
        .member-item { display: flex; align-items: center; justify-content: space-between; padding: 12px; border-bottom: 1px solid #eee; }
        .member-info { display: flex; align-items: center; gap: 15px; }
        .member-avatar { width: 40px; height: 40px; border-radius: 50%; background: #2196F3; color: white; display: flex; align-items: center; justify-content: center; font-weight: bold; }
        .member-name { font-weight: bold; }
        .member-role { font-size: 12px; color: white; padding: 2px 8px; border-radius: 10px; }
        .role-yönetici { background: #2196F3; }
        .role-üye { background: #4CAF50; }
        .role-gözlemci { background: #ff9800; }
        
        .actions { display: flex; gap: 10px; margin-top: 20px; }
        .btn { padding: 10px 20px; border-radius: 5px; text-decoration: none; display: inline-flex; align-items: center; gap: 8px; }
        .btn-primary { background: #2196F3; color: white; }
        .btn-success { background: #4CAF50; color: white; }
        .btn-secondary { background: #666; color: white; }
        .btn:hover { opacity: 0.9; }
        
        .task-status { display: inline-block; padding: 3px 10px; border-radius: 12px; font-size: 12px; }
        .status-yapılacak { background: #ffeb3b; color: #333; }
        .status-yapılıyor { background: #2196F3; color: white; }
        .status-incelemede { background: #ff9800; color: white; }
        .status-tamamlandı { background: #4CAF50; color: white; }
        
        .comment-bubble { background: #f0f8ff; padding: 10px; border-radius: 10px; margin-bottom: 10px; }
        .comment-author { font-weight: bold; color: #2196F3; }
        .comment-date { font-size: 12px; color: #666; }
        
        @media (max-width: 1024px) {
            .content-grid { grid-template-columns: 1fr; }
        }
        
        @media (max-width: 768px) {
            .stats-grid { grid-template-columns: repeat(2, 1fr); }
            .project-meta { flex-wrap: wrap; }
        }
    </style>
</head>
<body>
    <div class="header">
        <div class="header-content">
            <div class="logo">ProjeYönetim</div>
            <div class="nav-links">
                <a href="dashboard.php">Dashboard</a>
                <a href="projects.php">Projeler</a>
                <a href="tasks.php">Görevler</a>
            </div>
        </div>
    </div>
    
    <div class="container">
        <!-- Proje Başlık -->
        <div class="project-header">
            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 20px;">
                <div>
                    <h1 class="project-title"><?php echo htmlspecialchars($project['project_name']); ?></h1>
                    <div style="display: flex; gap: 10px; margin-bottom: 15px;">
                        <span class="status-badge status-<?php echo strtolower(str_replace(' ', '-', $project['status'])); ?>">
                            <?php echo $project['status']; ?>
                        </span>
                        <span class="priority-badge priority-<?php echo strtolower($project['priority']); ?>">
                            <?php echo $project['priority']; ?>
                        </span>
                    </div>
                </div>
                <div class="actions">
                    <a href="add_task.php?project_id=<?php echo $project_id; ?>" class="btn btn-primary">+ Görev Ekle</a>
                    <a href="team.php?project_id=<?php echo $project_id; ?>" class="btn btn-success">Ekip Yönet</a>
                </div>
            </div>
            
            <div class="project-meta">
                <div class="meta-item">
                    <strong>Başlangıç:</strong> <?php echo $project['start_date']; ?>
                </div>
                <div class="meta-item">
                    <strong>Bitiş:</strong> <?php echo $project['end_date']; ?>
                </div>
                <div class="meta-item">
                    <strong>Kategori:</strong> <?php echo $project['category']; ?>
                </div>
                <div class="meta-item">
                    <strong>Oluşturan:</strong> <?php echo htmlspecialchars($project['creator_name']); ?>
                </div>
            </div>
            
            <div class="project-description">
                <h4 style="margin-bottom: 10px;">Açıklama</h4>
                <?php echo nl2br(htmlspecialchars($project['description'])); ?>
            </div>
        </div>
        
        <!-- İstatistikler -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['total_tasks'] ?? 0; ?></div>
                <div class="stat-label">Toplam Görev</div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['completed_tasks'] ?? 0; ?></div>
                <div class="stat-label">Tamamlanan</div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['overdue_tasks'] ?? 0; ?></div>
                <div class="stat-label">Geciken</div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?php echo $progress; ?>%</div>
                <div class="stat-label">İlerleme</div>
                <div class="progress-bar">
                    <div class="progress-fill" style="width: <?php echo $progress; ?>%"></div>
                </div>
            </div>
        </div>
        
        <!-- Ana İçerik -->
        <div class="content-grid">
            <!-- Sol Kolon: Görevler -->
            <div class="section">
                <h3>Görevler</h3>
                
                <?php if ($tasks->num_rows > 0): ?>
                <table class="tasks-table">
                    <thead>
                        <tr>
                            <th>Görev Adı</th>
                            <th>Durum</th>
                            <th>Atanan</th>
                            <th>Son Tarih</th>
                            <th>İşlem</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while($task = $tasks->fetch_assoc()): ?>
                        <tr>
                            <td>
                                <strong><?php echo htmlspecialchars($task['task_name']); ?></strong><br>
                                <small style="color: #666;"><?php echo substr($task['description'], 0, 50); ?>...</small>
                            </td>
                            <td>
                                <span class="task-status status-<?php echo strtolower($task['status']); ?>">
                                    <?php echo $task['status']; ?>
                                </span>
                            </td>
                            <td><?php echo $task['assignee_name'] ?? 'Atanmadı'; ?></td>
                            <td>
                                <?php 
                                $end_date = new DateTime($task['end_date']);
                                $today = new DateTime();
                                $days_diff = $today->diff($end_date)->days;
                                
                                if ($task['status'] == 'Tamamlandı') {
                                    echo '<span style="color: #4CAF50;">' . $task['end_date'] . '</span>';
                                } elseif ($end_date < $today) {
                                    echo '<span style="color: #f44336; font-weight: bold;">' . $task['end_date'] . ' (Gecikti)</span>';
                                } elseif ($days_diff <= 3) {
                                    echo '<span style="color: #ff9800; font-weight: bold;">' . $task['end_date'] . ' (Yaklaşıyor)</span>';
                                } else {
                                    echo $task['end_date'];
                                }
                                ?>
                            </td>
                            <td>
                                <a href="task_detail.php?id=<?php echo $task['id']; ?>" style="color: #2196F3; text-decoration: none;">Detay</a>
                            </td>
                        </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
                <?php else: ?>
                <div style="text-align: center; padding: 40px; color: #666;">
                    <p>Henüz görev eklenmemiş.</p>
                    <a href="add_task.php?project_id=<?php echo $project_id; ?>" class="btn btn-primary">İlk Görevi Ekle</a>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Sağ Kolon: Ekip ve Aktivite -->
            <div style="display: flex; flex-direction: column; gap: 30px;">
                <!-- Ekip Üyeleri -->
                <div class="section">
                    <h3>Ekip Üyeleri</h3>
                    
                    <ul class="members-list">
                        <?php while($member = $members->fetch_assoc()): 
                            $initials = getInitials($member['full_name']);
                            $role_class = strtolower($member['role']);
                        ?>
                        <li class="member-item">
                            <div class="member-info">
                                <div class="member-avatar"><?php echo $initials; ?></div>
                                <div>
                                    <div class="member-name"><?php echo htmlspecialchars($member['full_name']); ?></div>
                                    <div style="font-size: 14px; color: #666;"><?php echo $member['email']; ?></div>
                                </div>
                            </div>
                            <span class="member-role role-<?php echo $role_class; ?>"><?php echo $member['role']; ?></span>
                        </li>
                        <?php endwhile; ?>
                    </ul>
                    
                    <div style="margin-top: 20px;">
                        <a href="team.php?project_id=<?php echo $project_id; ?>" class="btn btn-secondary" style="width: 100%; text-align: center;">Ekip Yönetimi</a>
                    </div>
                </div>
                
                <!-- Hızlı İşlemler -->
                <div class="section">
                    <h3>Hızlı İşlemler</h3>
                    <div style="display: flex; flex-direction: column; gap: 10px;">
                        <a href="add_task.php?project_id=<?php echo $project_id; ?>" class="btn btn-primary">Yeni Görev Ekle</a>
                        <a href="team.php?project_id=<?php echo $project_id; ?>" class="btn btn-success">Ekip Üyesi Ekle</a>
                        <a href="edit_project.php?id=<?php echo $project_id; ?>" class="btn btn-secondary">Projeyi Düzenle</a>
                        <a href="reports.php?project_id=<?php echo $project_id; ?>" class="btn btn-secondary">Rapor Al</a>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <?php
    // İsimden baş harfleri al
    function getInitials($name) {
        $words = explode(' ', $name);
        $initials = '';
        foreach ($words as $word) {
            $initials .= strtoupper(substr($word, 0, 1));
        }
        return substr($initials, 0, 2);
    }
    ?>
</body>
</html>