<?php
declare(strict_types=1);

// Basic helpers for a tiny JSON-backed app.

function app_root(): string {
  return dirname(__DIR__);
}

function data_path(string $file): string {
  return app_root() . '/data/' . ltrim($file, '/');
}

function ensure_data_dir(): void {
  $dir = app_root() . '/data';
  if (!is_dir($dir)) {
    mkdir($dir, 0775, true);
  }
}

function json_read(string $file, mixed $default): mixed {
  ensure_data_dir();
  $path = data_path($file);
  if (!is_file($path)) return $default;

  $raw = file_get_contents($path);
  if ($raw === false || trim($raw) === '') return $default;

  $decoded = json_decode($raw, true);
  return (json_last_error() === JSON_ERROR_NONE) ? $decoded : $default;
}

function json_write(string $file, mixed $data): void {
  ensure_data_dir();
  $path = data_path($file);
  $tmp = $path . '.tmp';
  $json = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
  if ($json === false) {
    throw new RuntimeException('JSON encode failed: ' . json_last_error_msg());
  }
  file_put_contents($tmp, $json, LOCK_EX);
  rename($tmp, $path);
}

function e(string $s): string {
  return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
}

function session_start_safe(): void {
  if (session_status() !== PHP_SESSION_ACTIVE) {
    session_start();
  }
}

function csrf_token(): string {
  session_start_safe();
  if (empty($_SESSION['csrf'])) {
    $_SESSION['csrf'] = bin2hex(random_bytes(16));
  }
  return (string)$_SESSION['csrf'];
}

function csrf_or_fail(): void {
  session_start_safe();
  $t = $_POST['csrf'] ?? $_GET['csrf'] ?? '';
  if (!$t || !hash_equals((string)($_SESSION['csrf'] ?? ''), (string)$t)) {
    http_response_code(403);
    echo 'CSRF doğrulaması başarısız.';
    exit;
  }
}

function redirect(string $to): never {
  header('Location: ' . $to);
  exit;
}

function pad_code(string $value, int $len): string {
  $v = trim($value);
  // If numeric, left-pad with zeros.
  if ($v !== '' && ctype_digit($v)) {
    return str_pad($v, $len, '0', STR_PAD_LEFT);
  }
  // Otherwise keep as-is (e.g., AS).
  return $v;
}

function lists_default(): array {
  return [
    'categories' => [
      ['code' => '007', 'label' => 'Örnek Kategori'],
    ],
    'types' => [
      ['code' => 'AS', 'label' => 'Al-Sat'],
      ['code' => 'UR', 'label' => 'Üretim'],
    ],
    'groups' => [
      ['code' => '501', 'label' => 'Örnek Grup'],
    ],
    'reyon' => [
      ['code' => '009', 'label' => 'Örnek Reyon'],
    ],
  ];
}

function get_lists(): array {
  $lists = json_read('lists.json', null);
  if (!is_array($lists)) {
    $lists = lists_default();
    json_write('lists.json', $lists);
  }

  // Normalize keys.
  foreach (['categories','types','groups','reyon'] as $k) {
    if (!isset($lists[$k]) || !is_array($lists[$k])) $lists[$k] = [];
  }
  return $lists;
}

function save_lists(array $lists): void {
  json_write('lists.json', $lists);
}

function get_counters(): array {
  $c = json_read('counters.json', []);
  return is_array($c) ? $c : [];
}

function save_counters(array $c): void {
  json_write('counters.json', $c);
}

function get_items(): array {
  $items = json_read('items.json', []);
  $items = is_array($items) ? $items : [];

  // Backward-compatible: older items may not have an id.
  $changed = false;
  foreach ($items as $i => $it) {
    if (!is_array($it)) {
      unset($items[$i]);
      $changed = true;
      continue;
    }
    if (empty($it['id'])) {
      $items[$i]['id'] = bin2hex(random_bytes(8));
      $changed = true;
    }
  }
  // Reindex after possible unsets.
  $items = array_values($items);
  if ($changed) {
    save_items($items);
  }

  return $items;
}

function save_items(array $items): void {
  json_write('items.json', $items);
}

function find_label(array $list, string $code): string {
  foreach ($list as $row) {
    if ((string)($row['code'] ?? '') === $code) return (string)($row['label'] ?? '');
  }
  return '';
}

function now_tr(): string {
  // Simple ISO-like timestamp.
  return date('Y-m-d H:i:s');
}
