(function () {
  const root = document.documentElement;
  const THEME_KEY = 'sk-theme';

  function applyTheme(theme) {
    if (theme === 'dark') {
      root.classList.add('dark');
      document.body.classList.add('bg-gray-950', 'text-gray-100');
      document.body.classList.remove('bg-gray-50', 'text-gray-900');
    } else {
      root.classList.remove('dark');
      document.body.classList.add('bg-gray-50', 'text-gray-900');
      document.body.classList.remove('bg-gray-950', 'text-gray-100');
    }
  }

  function initTheme() {
    const saved = localStorage.getItem(THEME_KEY);
    applyTheme(saved === 'dark' ? 'dark' : 'light');
  }

  function toggleTheme() {
    const isDark = root.classList.contains('dark');
    const next = isDark ? 'light' : 'dark';
    localStorage.setItem(THEME_KEY, next);
    applyTheme(next);
  }

  async function copyText(text) {
    try {
      await navigator.clipboard.writeText(text);
      toast('Kopyalandı');
    } catch (e) {
      // Fallback
      const ta = document.createElement('textarea');
      ta.value = text;
      document.body.appendChild(ta);
      ta.select();
      document.execCommand('copy');
      ta.remove();
      toast('Kopyalandı');
    }
  }

  let toastTimer = null;
  function toast(msg) {
    let el = document.getElementById('skToast');
    if (!el) {
      el = document.createElement('div');
      el.id = 'skToast';
      el.className = 'fixed bottom-4 left-1/2 -translate-x-1/2 rounded-2xl border border-gray-200 bg-white px-4 py-2 text-sm shadow';
      document.body.appendChild(el);
    }
    el.textContent = msg;
    el.style.display = 'block';
    clearTimeout(toastTimer);
    toastTimer = setTimeout(() => {
      el.style.display = 'none';
    }, 1400);
  }

  function bindCopyButtons() {
    document.addEventListener('click', (ev) => {
      const t = ev.target;
      const btn = t && t.closest ? t.closest('[data-copy]') : null;
      if (!btn) return;
      const text = btn.getAttribute('data-copy') || '';
      if (text) copyText(text);
    });

    const copyPreview = document.getElementById('copyPreview');
    if (copyPreview) {
      copyPreview.addEventListener('click', () => {
        const preview = document.getElementById('previewCode');
        if (preview) copyText(preview.textContent.trim());
      });
    }
  }

  function bindThemeButton() {
    const btn = document.getElementById('themeBtn');
    if (!btn) return;
    btn.addEventListener('click', toggleTheme);
  }

  function bindSelectReload() {
    const ids = ['category', 'type', 'group', 'reyon'];
    const els = ids.map((id) => document.getElementById(id)).filter(Boolean);
    if (!els.length) return;

    els.forEach((el) => {
      el.addEventListener('change', () => {
        const url = new URL(window.location.href);
        ids.forEach((id) => {
          const sel = document.getElementById(id);
          if (sel) url.searchParams.set(id, sel.value);
        });
        // Keep page clean
        url.hash = '';
        window.location.href = url.toString();
      });
    });
  }

  initTheme();
  bindThemeButton();
  bindCopyButtons();
  bindSelectReload();
})();
